import { camelCase, cleanComponentProperties } from "../cleanFunc";
import { metaDataDrawer } from "../metadata";
import { setCustomNLSKey, setIgnoreKey } from "../pluginData";
import { formDataFunc, getNodeIdFunc, setNodeIdFunc, updateFormData } from "../processFunc";

export function generateTextCustomAttribute(node: any) {
    let childNode: any;
    if (node.type === 'TEXT') {
        childNode = node;
    } else {
        const textType = node?.componentProperties['Type']?.value;
        childNode = node?.children?.[0];

        if (textType !== 'Custom' || !childNode) return;
    }

    const textStyle = figma.getStyleById(childNode.textStyleId)?.name;
    const headings = ['H1', 'H2', 'H3', 'H4', 'H5', 'H6'];
    const property = textStyle && !headings.includes(textStyle) ? transformTypography(textStyle) : '';
    const textColorStyle = figma.getStyleById(childNode.fillStyleId)?.name;
    const colorProperty = textColorStyle ? `oj-text-color-${textColorStyle.split('/')[2].toLowerCase()}` : '';

    if (!property && !colorProperty) return;

    return [{
        colon: true,
        attribute: "class",
        value: `'${[property, colorProperty].filter(Boolean).join(' ')}'`
    }];
}

function transformTypography(input: string): string {
    const parts = input.split('/');
    const type = parts[0].toLowerCase();

    if (type === 'body') {
        const weight = parts[1].toLowerCase();
        const size = parts[2].toLowerCase();
        switch (weight) {
            case 'bold':
                return `oj-typography-body-${size} oj-typography-bold`;
            case 'semibold':
                return `oj-typography-body-${size} oj-typography-semi-bold`;
            case 'regular':
                return `oj-typography-body-${size}`;
            default:
                throw new Error("Invalid style for Body type");
        }
    } else {
        const size = parts[1].toLowerCase();
        return `oj-typography-${type}-${size}`;
    }
}

export function addDrawer(node: any) {
    const linkReactionResult = linkReaction(node, formDataFunc());
    let keyName: any;
    let mainFormData: any;
    if (linkReactionResult) {
        ({ keyName, mainFormData } = linkReactionResult);
        updateFormData(mainFormData);
    }

    if (keyName) {
        const hookFunction = {
            code: `Navigation.openRightPanel('${formDataFunc().componentName}-drawer-content',{},self.nls.${keyName});`
        }
        return { hookFunction }
    }
    return;
}

export function linkReaction(node: any, formData: any) {
    let destId = '';
    for (let i of node.reactions) {
        if (i.trigger.type === 'ON_CLICK') {
            destId = i.action.destinationId;
            break;
        }
    }
    const destNode: any = figma.getNodeById(destId);
    if (!destNode || destNode.name !== 'Drawer') {
        return null;
    }

    const output = cleanComponentProperties(destNode.componentProperties);
    const value = output['headertext'] || '';
    const keyName = camelCase(value);
    const foundNode = destNode.findOne((n: any) => n.name === 'Container');
    if (foundNode) {
        const projectForm = {
            componentType: "SINGLE_PAGE",
            componentName: `${formData.componentName}-drawer-content`,
            componentDescription: `${formData.componentDescription} Drawer Content`,
            usableComponent: false,
            moduleName: formData.moduleName || formData.componentName,
            moduleDescription: formData.moduleDescription,
            trunkPath: formData.trunkPath,
            menuEntry: false,
            readOnly: false
        }
        const nodeId = getNodeIdFunc();
        const drawerMetaData = metaDataDrawer(foundNode.children[0], projectForm);
        setNodeIdFunc(nodeId);
        figma.ui.postMessage({ drawerFormData: projectForm, drawerMetaData });
    }

    setIgnoreKey(destNode.id);
    setCustomNLSKey({
        [keyName]: value
    })
    return { keyName, mainFormData: formData };
}

export function textTagTypeFunc(node: any): string {
    let childNode: any;
    if (node.type === 'TEXT') {
        childNode = node;
    } else {
        childNode = node?.children?.[0];
    }
    const textStyle = figma.getStyleById(childNode.textStyleId)?.name;
    let tagType = 'span';
    if (textStyle) {
        switch (textStyle) {
            case 'H1':
            case 'H2':
                tagType = 'H2';
                break;
            case 'H3':
                tagType = 'H3';
                break;
            case 'H4':
                tagType = 'H4';
                break;
            case 'H5':
                tagType = 'H5';
                break;
            case 'Label':
                tagType = 'oj-label';
                break;
            default:
                tagType = node.parent.type === 'FRAME' && node.parent.layoutMode === 'HORIZONTAL'
                    ? 'span'
                    : 'div';
                break;
        }
    } else {
        tagType = node.parent.type === 'FRAME' && node.parent.layoutMode === 'HORIZONTAL'
            ? 'span'
            : 'div';
    }

    return tagType;
}
