import { useState, useEffect } from "preact/hooks";
import { useAppContext } from '../../context/AppContext';
import { useAppDataContext } from "../../context/PluginData";
import ArrayDataProvider = require("ojs/ojarraydataprovider");
import "ojs/ojselectsingle";
import "ojs/ojbutton";
import ContainerComponent from "../utility/ContainerComponent";
import Header from "../utility/Header";
import ButtonComponent from "../utility/Button";
import centreIcon from "../images/vertical-align-center.svg"

interface GridOption {
    value: number;
    key: string;
    alignItems?: string;
    justifyContent?: string;
}

export function AddGrid() {

    const { setType } = useAppContext();
    const { pluginData } = useAppDataContext();
    const gridComponent = pluginData?.gridComponent;
    const selectedComponent = pluginData?.selectedGridComponent;
    const [gridMap, setGridMap] = useState<GridOption[]>(() =>
        gridComponent.map((key: string) => ({
            key,
            value: 1,
            alignItems: 'center',
            justifyContent: 'center'
        }))
    );

    useEffect(() => {
        if (pluginData?.fetchGridData?.length > 0) {
            try {
                setGridMap(pluginData?.fetchGridData);
            } catch (error) {
                console.error("Error parsing conditionData:", error);
                setGridMap(gridComponent.map((key: string) => ({
                    key,
                    value: 1,
                    alignItems: 'center',
                    justifyContent: 'center'
                })));
            }
        } else {
            setGridMap(gridComponent.map((key: string) => ({
                key,
                value: 1,
                alignItems: 'center',
                justifyContent: 'center'
            })));
        }
    }, [pluginData, gridComponent]);

    const gridOptions = Array.from({ length: 12 }, (_, index) => {
        const value = index + 1;
        return { value, label: value };
    });

    const gridOptionsArrayProvider = new ArrayDataProvider(gridOptions, {
        keyAttributes: "value",
    })

    const handleSave = () => {
        const sum = gridMap.reduce((accumulator, current) => accumulator + current.value, 0);
        if (sum !== 12) {
            return;
        } else {
            parent.postMessage({ pluginMessage: { type: 'save-grid-data', gridMap, selectedComponent } }, '*');
        }
        setType('gridSuccess');
    }

    const handleBack = () => {
        setType('generate')
        parent.postMessage({ pluginMessage: { type: 'generate' } }, '*');
    }

    const handleGridOptions = (index: number, event: any) => {
        const newValue = event.detail.value;
        setGridMap((prevGridMap: GridOption[]) => {
            const updated = prevGridMap.map((item, idx) =>
                idx === index ? { ...item, value: newValue } : item
            );

            if (prevGridMap[index].value === newValue) {
                return prevGridMap;
            }

            return updated;
        });
    };

    const handleAlignItems = (index: number, event: any) => {
        const newValue = event.detail.value;
        setGridMap((prevGridMap: GridOption[]) => {
            const updated = prevGridMap.map((item, idx) =>
                idx === index ? { ...item, alignItems: newValue } : item
            );

            if (prevGridMap[index].alignItems === newValue) {
                return prevGridMap;
            }

            return updated;
        });
    };

    const handleJustifyContent = (index: number, event: any) => {
        const newValue = event.detail.value;
        setGridMap((prevGridMap: GridOption[]) => {
            const updated = prevGridMap.map((item, idx) =>
                idx === index ? { ...item, justifyContent: newValue } : item
            );

            if (prevGridMap[index].justifyContent === newValue) {
                return prevGridMap;
            }

            return updated;
        });
    };

    return (
        <>
            <ContainerComponent>
                <Header title='Add Grid' handleBack={handleBack} />

                <div style={{ overflowY: 'auto', flex: 1, marginBottom: '25px' }}>
                    {gridMap?.map((child: any, index: number) => (
                        <div style={{ display: 'flex', flexDirection: 'column', marginBottom: '24px', marginRight: '10px' }}>
                            <div slot="secondary" class="oj-typography-body-md oj-typography-semi-bold" style={{ marginBottom: '10px' }}>
                                {child.key}
                            </div>
                            <div style={{ width: '100%', marginBottom: '10px' }}>
                                <oj-select-single
                                    id="gridSelect"
                                    label-hint="Grid Size"
                                    label-edge="inside"
                                    class="oj-form-control-max-width-md"
                                    data={gridOptionsArrayProvider}
                                    value={child.value}
                                    onvalueChanged={(event) => handleGridOptions(index, event)}>
                                </oj-select-single>
                            </div>
                            <div style={{ display: 'flex', alignItems: 'flex-start', justifyContent: 'space-between' }}>
                                <div style={{ flex: 1 }}>
                                    <oj-buttonset-one
                                        id="alignItems"
                                        value={gridMap[index].alignItems}
                                        onvalueChanged={(event) => handleAlignItems(index, event)}
                                        class="oj-button-sm oj-sm-margin-1x-bottom"
                                        display="icons">
                                        <oj-option value='start'>
                                            <span slot="startIcon" class="oj-ux-ico-layout-align-left"></span>
                                            <span>Left</span>
                                        </oj-option>
                                        <oj-option value='center'>
                                            <span slot="startIcon" class="oj-ux-ico-layout-align-v"></span>
                                            <span>Center</span>
                                        </oj-option>
                                        <oj-option value='end'>
                                            <span slot="startIcon" class="oj-ux-ico-layout-align-right"></span>
                                            <span>Right</span>
                                        </oj-option>
                                    </oj-buttonset-one>
                                </div>
                                <div style={{ flex: 1, display: 'flex', justifyContent: 'flex-end' }}>
                                    <oj-buttonset-one
                                        id="justifyContent"
                                        value={gridMap[index].justifyContent}
                                        onvalueChanged={(event) => handleJustifyContent(index, event)}
                                        class="oj-button-sm oj-sm-margin-1x-bottom"
                                        display="icons">
                                        <oj-option value='start'>
                                            <span slot="startIcon" class="oj-ux-ico-vertical-align-top"></span>
                                            <span>Top</span>
                                        </oj-option>
                                        <oj-option value='center'>
                                            <span slot="startIcon">
                                                <img src={centreIcon} />
                                            </span>
                                            <span>Center</span>
                                        </oj-option>
                                        <oj-option value='end'>
                                            <span slot="startIcon" class="oj-ux-ico-vertical-align-bottom"></span>
                                            <span>Bottom</span>
                                        </oj-option>
                                    </oj-buttonset-one>
                                </div>
                            </div>
                        </div>

                    ))}
                    {/* {gridError && (
                        <p style={{ color: 'red', fontSize: '14px', fontWeight: 'bold', marginTop: '10px', display: 'flex', alignItems: 'center' }}>
                            <span class="oj-ux-ico-warning oj-ux-icon-size-5x" style={{ marginRight: "5px" }} role="img" aria-label="warning"></span>
                            Grid should sum upto 12
                        </p>
                    )} */}

                </div>

                <ButtonComponent label='Save' onClick={handleSave} disabled={false} />

            </ContainerComponent>
        </>
    )
}