import { useState, useEffect } from 'preact/hooks';
import { useAppContext } from '../../context/AppContext';
import Tree from "./tree";
import ContainerComponent from '../utility/ContainerComponent';
import Header from '../utility/Header';
import ButtonComponent from '../utility/Button';

interface formProps {
    setMenuData: (type: any) => void;
    menuData: any;
}

interface Folder {
    name: string,
    fid: string
}

interface MenuContainer {
    name: string;
    data: any;
}

export function MenuEntry(props: formProps) {

    const { setType, formData, setFormData } = useAppContext();
    const [folderSelected, setFolderSelected] = useState<Folder>();
    const { menuData, setMenuData } = props;
    const [currentChildIndex, setCurrentChildIndex] = useState(0);
    const [currentChildData, setCurrentChildData] = useState<MenuContainer>({ name: '', data: [] });

    async function updateData() {
        try {
            const response = await fetch('http://localhost:3000/files/treeDisplay', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    "userType": formData.userType
                })
            });
            if (!response.ok) {
                throw new Error(`HTTP error! Status: ${response.status}`);
            } else {
                const result = await response.json();
                const menuContainer: any[] = result.map((subArray: any) => ({
                    name: subArray.name,
                    data: [{
                        name: 'menu',
                        menuMappedValue: `Menu - ${subArray.name}`,
                        submenus: subArray.menuData
                    }]
                }));
                setMenuData(menuContainer);
                setCurrentChildData(menuContainer[0]);
            }
        } catch (error) {
            console.error('Error in written metadata API', error);
        }
    }

    useEffect(() => {
        updateData();
    }, []);

    useEffect(() => {
        if (menuData.length > 0) {
            setCurrentChildData(menuData[currentChildIndex]);
        }
    }, [currentChildIndex, menuData]);

    function addNewData(data: any, path: any, newData: any): any[] {
        const pathComponents = typeof path === 'string' && path.includes('/') ? path.split('/') : (path ? [path] : ['menu']);

        let currentLevel: any[] = data;
        for (const component of pathComponents) {
            let found = false;
            for (const submenu of currentLevel) {
                if (submenu.name.toLowerCase() === component) {
                    currentLevel = submenu.submenus || [];
                    submenu.submenus = currentLevel;
                    found = true;
                    break;
                }
            }
            if (!found) {
                throw new Error(`Path component '${component}' not found in the data structure`);
            }
        }

        currentLevel.push(newData);

        return data;
    }

    function getTypeFromDescription(description: string): string {
        const words = description.trim().split(/\s+/);
        if (words.length === 1) {
            return words[0].toUpperCase();
        }
        return words.map(word => word[0]).join('').toUpperCase();
    }

    const handleSaveAndProceed = () => {
        let newData: any = {
            ...(formData.componentName !== '' && { name: formData.componentName }),
            ...(formData.componentType === 'FLOW' && { class: 'flow' }),
            ...(formData.moduleName !== '' && { module: formData.moduleName })
        };

        if(formData.componentType === 'OVERVIEW'){
            newData.type = "TEMPLATE",
            newData.templateMode = "OVERVIEW",
            newData.templateTab = getTypeFromDescription(formData.componentDescription)
        }

        const menuCopy = [...currentChildData.data];
        const updatedData = addNewData(menuCopy, folderSelected?.fid, newData);
        setMenuData((prevMenuData: any) => (
            prevMenuData.map((item: any) => {
                if (item.name === currentChildData.name) {
                    return { ...item, data: updatedData };
                }
                return item;
            })
        ));
        if (currentChildIndex < menuData.length - 1) {
            setCurrentChildIndex(currentChildIndex + 1);
        } else {
            if (formData.componentType === 'SINGLE_PAGE' && formData.usableComponent) {
                setFormData({
                    ...formData
                });
            }
            parent.postMessage({ pluginMessage: { type: 'export-proceed', formData } }, '*');
            setType('exportProceed');
        }
    };

    const handleBack = () => {
        setFormData({
            ...formData,
            menuEntry: false
        });
        setType("export");
    }

    return (
        <>
            <ContainerComponent>
                <Header title='Menu Entry' handleBack={handleBack} />
                <oj-input-text
                    id="componentName"
                    value={formData.componentName}
                    label-hint="Component Name"
                    label-edge="inside"
                    readonly
                    class="oj-sm-margin-3x-bottom"
                ></oj-input-text>
                {/* display error message is tree data not fetched */}
                <div style={{ overflowY: 'auto', flex: 1, marginBottom: '25px' }}>
                    <Tree data={currentChildData.data} folderSelected={folderSelected} setFolderSelected={setFolderSelected} />
                </div>
                <ButtonComponent
                    label='Save And Proceed'
                    onClick={handleSaveAndProceed}
                    disabled={false}
                />
            </ContainerComponent>
        </>
    )
}