import { FunctionComponent } from 'preact';
import { useState } from 'preact/hooks';

interface MenuItem {
    name: string;
    menuMappedValue: string;
    submenus?: MenuItem[];
    fid?: string;
}

interface CollapsibleListProps {
    data: MenuItem[];
    setType?: (type: string) => void;
    setFolderSelected: (type: any) => void;
    folderSelected: any;
}

interface CollapsibleListItemProps {
    item: MenuItem;
    setFolderSelected: (type: any) => void;
    folderSelected: any;
}

const Tree: FunctionComponent<CollapsibleListProps> = ({ data, folderSelected, setFolderSelected }) => {


    const renderList = (items: any[]) => {
        return items.map(item => (
            <CollapsibleListItem key={item.name} item={item} folderSelected={folderSelected} setFolderSelected={setFolderSelected} />
        ));
    };


    return (
        <ul style={styles.collapsibleList} >
            {renderList(data)}
        </ul>
    );
};

const CollapsibleListItem: FunctionComponent<CollapsibleListItemProps> = ({ item, folderSelected, setFolderSelected }) => {
    const [isOpen, setIsOpen] = useState(false);

    const hasSubmenus = item.submenus && item.submenus.length > 0;

    const toggleOpen = () => {
        setIsOpen(!isOpen);
        if (hasSubmenus) {
            if (item?.fid)
                setFolderSelected({ name: item.name, fid: item.fid });
            else
                setFolderSelected({ name: 'menu', fid: item.fid })
        }
    };

    const isSelected = folderSelected && folderSelected.name === item.name;

    return (
        <>
        {item?.menuMappedValue && (
            <li>
                {/* color in color file */}
                <div style={{ ...styles.collapsibleListItem, backgroundColor: isSelected ? 'rgba(135, 206, 235, 0.3)' : 'transparent', border: isSelected ? '1px solid rgba(135, 206, 235)' : 'none', padding: '5px', whiteSpace: 'nowrap' }} onClick={toggleOpen}>
                    {hasSubmenus && (
                        <>
                            <span
                                style={{ ...styles.arrow, ...(isOpen ? styles.arrowOpen : {}) }}
                                class={`oj-ux-ico-arrow-circle-right oj-ux-icon-size-5x`}
                                role="img"
                                aria-label="arrow"
                            ></span>
                            <span class="oj-typography-body-md">{item.menuMappedValue}</span>
                        </>
                    )}
                    {!hasSubmenus && (

                        <li style={styles.leafItem}>
                            <span
                                style={{ ...styles.arrow, ...(isOpen ? styles.arrowOpen : {}) }}
                                class={`oj-ux-ico-file-doc-alt oj-ux-icon-size-5x`}
                                role="img"
                                aria-label="arrow"
                            ></span>
                            <span class="oj-typography-body-md">{item.menuMappedValue}</span>
                        </li>
                    )}
                </div>
                {hasSubmenus && isOpen && (
                    <ul style={{ ...styles.collapsibleContent, ...(isOpen ? styles.collapsibleContentOpen : {}) }}>
                        {item.submenus!.map((submenu, index) => (
                            <CollapsibleListItem key={index} item={submenu} folderSelected={folderSelected} setFolderSelected={setFolderSelected} />
                        ))}
                    </ul>
                )}
            </li>
        )}
        </>

    );
};

const styles = {
    collapsibleList: {
        listStyleType: 'none',
        paddingLeft: '10px'
    },
    collapsibleListItem: {
        cursor: 'pointer',
        display: 'flex',
        alignItems: 'center',
    },
    leafItem: {
        listStyleType: 'none',
        paddingLeft: '20px',
        whiteSpace: 'nowrap',
    },
    arrow: {
        display: 'inline-block',
        transition: 'transform 0.5s',
        marginRight: '5px',
    },
    arrowOpen: {
        transform: 'rotate(90deg)',
    },
    collapsibleContent: {
        listStyleType: 'none',
        maxHeight: '0',
        overflow: 'hidden',
        transition: 'max-height 0.5s ease-out',
        paddingRight: '10px'
    },
    collapsibleContentOpen: {
        maxHeight: '3000px',
        transition: 'max-height 0.4s ease-in',
        marginBottom: '10px'
    },
};

export default Tree;
