import { generate } from "../cleanFunc";
import { fetchIconClass, mapAvatarSize } from "../components/avatar";
import { buttonIconProperties, extractTextForButton } from "../components/button";
import { chartCustomAttributes, fetchChartType, slotChildren, slotsFunc, templateSlots } from "../components/chart";
import { generateCheckBoxOrientation, generateCheckBoxProperties } from "../components/checkBox";
import { extractTextFromNode, generateMappedValue } from "../components/commonFunc";
import { generateFileChildren } from "../components/filePicker";
import { generateSelectSingleExtraProperties, generateSelectSingleOptionList, inputReactions, searchPlaceholder } from "../components/inputs";
import { pageSectionMapping } from "../components/pageSection";
import { generateRadioOrientation, generateRadioSetProperties, radioConnectedNodes } from "../components/radioSet";
import { extractMinMax } from "../components/slider";
import { generateCustomAttribute, generateTabBarProperties, generateTabBarResult, tabBarConnectedNodes } from "../components/tabBar";
import { generateTableChildren, generateTableProperties } from "../components/table";
import { addDrawer, generateTextCustomAttribute, textTagTypeFunc } from "../components/textLink";
import { generateToggleButtonProperties, toggleConnectedNodes } from "../components/toggleButton";
import { traverseNode } from "../traverseNode";

type CustomAttributeRule = {
    value: string | boolean | ((node: any) => string | boolean);
    attrKey: string;
    attrValue: string | ((node: any) => string);
};

// Type for the customAttribute structure within each component
type CustomAttributeMap = {
    [property: string]: CustomAttributeRule[];
};

type MapStructure = {
    [key: string]: {
        properties: {
            [key: string]: {
                keyName: string;
                value?: number | string | ((value: any, node: any) => any);
            }
        };
        type: string;
        customAttribute?: CustomAttributeMap;
        extraProperties?: (node: any) => Record<string, any>;
        connectedNodes?: (node: any) => any[];
    }
};

export const componentMap: MapStructure = {
    inputText: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            labelPosition: {
                keyName: 'labelPosition'
            },
            state: {
                keyName: 'state'
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            }
        },
        type: 'InputBox',
        customAttribute: {
            placeholder: [
                {
                    value: (node: any) => {
                        const reaction = inputReactions(node);
                        return reaction !== '' ? reaction : false;
                    },
                    attrKey: "placeholder",
                    attrValue: (node: any) => {
                        const reaction = inputReactions(node);
                        return reaction !== '' ? `'${reaction}'` : '';
                    }
                }
            ]
        }
    },
    inputPassword: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            labelPosition: {
                keyName: 'labelPosition'
            },
            state: {
                keyName: 'state'
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            }
        },
        type: 'InputPassword',
        customAttribute: {
            viewButton: [
                {
                    value: true,
                    attrKey: 'mask-on',
                    attrValue: 'visible'
                }
            ],
            placeholder: [
                {
                    value: (node: any) => {
                        const reaction = inputReactions(node);
                        return reaction !== '' ? reaction : false;
                    },
                    attrKey: "placeholder",
                    attrValue: (node: any) => {
                        const reaction = inputReactions(node);
                        return reaction !== '' ? `'${reaction}'` : '';
                    }
                }
            ]
        }
    },
    inputNumber: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            labelPosition: {
                keyName: 'labelPosition'
            },
            state: {
                keyName: 'state'
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            }
        },
        type: 'InputNumber',
        customAttribute: {
            placeholder: [
                {
                    value: (node: any) => {
                        const reaction = inputReactions(node);
                        return reaction !== '' ? reaction : false;
                    },
                    attrKey: "placeholder",
                    attrValue: (node: any) => {
                        const reaction = inputReactions(node);
                        return reaction !== '' ? `'${reaction}'` : '';
                    }
                }
            ]
        }
    },
    inputTextArea: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            }
        },
        type: 'TextArea',
        customAttribute: {
            placeholder: [
                {
                    value: (node: any) => {
                        const reaction = inputReactions(node);
                        return reaction !== '' ? reaction : false;
                    },
                    attrKey: "placeholder",
                    attrValue: (node: any) => {
                        const reaction = inputReactions(node);
                        return reaction !== '' ? `'${reaction}'` : '';
                    }
                }
            ]
        }
    },
    inputDate: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            }
        },
        type: 'Date',
        customAttribute: {
            placeholder: [
                {
                    value: (node: any) => {
                        const reaction = inputReactions(node);
                        return reaction !== '' ? reaction : false;
                    },
                    attrKey: "placeholder",
                    attrValue: (node: any) => {
                        const reaction = inputReactions(node);
                        return reaction !== '' ? `'${reaction}'` : '';
                    }
                }
            ]
        }
    },
    inputSearch: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            customAttribute: {
                keyName: 'customAttribute',
                value: (_value: any, node: any) => searchPlaceholder(node)
            }
        },
        type: 'InputSearch',
        // customAttribute: {
        //     placeholder: [
        //         {
        //             value: (node: any) => {
        //                 const reaction = inputReactions(node);
        //                 return reaction !== '' ? reaction : false;
        //             },
        //             attrKey: "placeholder",
        //             attrValue: (node: any) => {
        //                 const reaction = inputReactions(node);
        //                 return reaction !== '' ? `'${reaction}'` : '';
        //             }
        //         }
        //     ]
        // }
    },
    pageSection: {
        properties: pageSectionMapping,
        type: 'Container',
        extraProperties: (node: any) => ({
            containerType: "PageSection",
            className: "Container",
        })
    },
    tabBar: {
        properties: {
            nameholder: { keyName: 'nameholder', value: () => "TabBar" },
            slots: { keyName: 'slots', value: () => [{ name: "itemTemplate" }] },
            customAttribute: {
                keyName: 'customAttribute',
                value: (_value: any, node: any) => [
                    { attribute: "class", value: `'${generateCustomAttribute(node)}'`, colon: true }
                ]
            },
            templateSlots: { keyName: 'templateSlots', value: () => [{ slotName: "itemTemplate", index: 1 }] },
            selection: {
                keyName: 'selection',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            edge: { keyName: 'edge', value: () => "'top'" },
            children: {
                keyName: 'children',
                value: (node: any, mappedName: string) => generateTabBarResult()
            }
        },
        type: 'TabBar',
        extraProperties: (node: any) => generateTabBarProperties(node),
        connectedNodes: (node: any) => tabBarConnectedNodes(node)

    },
    columnShare: {
        properties: {
            children: {
                keyName: 'children',
                value: (_value: any, node: any) =>
                    node.children
                        ?.flatMap((child: any) => traverseNode(child))
                        .filter((child: any) => Object.keys(child).length > 0) || []
            }
        },
        type: 'Container',
        extraProperties: () => ({
            containerType: "ColumnShare"
        })
    },
    inputSelectSingle: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            labelPosition: {
                keyName: 'labelPosition'
            },
            state: {
                keyName: 'state'
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            selectionType: {
                keyName: 'selectionType',
                value: 'One'
            },
            use: {
                keyName: 'use',
                value: 'ObservableVariable'
            },
            optionList: {
                keyName: 'optionList',
                value: (_value: any, node: any) => generateSelectSingleOptionList(node)
            }
        },
        type: "SelectOne",
        extraProperties: generateSelectSingleExtraProperties
    },
    switch: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            labelPosition: {
                keyName: 'labelPosition'
            },
            state: {
                keyName: 'state'
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            }
        },
        type: "Switch"
    },
    radioSet: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            orientation: {
                keyName: 'orientation',
                value: (_value: any, node: any) => generateRadioOrientation(node)
            },
            use: {
                keyName: 'use',
                value: 'ObservableVariable'
            },
        },
        type: 'RadioButton',
        extraProperties: (node: any) => generateRadioSetProperties(node),
        connectedNodes: (node: any) => radioConnectedNodes(node)
    },
    toggleButton: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            amount: {
                keyName: "amount",
            },
            nameholder: {
                keyName: 'nameholder',
                value: (() => `ButtonSetOne${generate()}`)()
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            use: {
                keyName: 'use',
                value: 'ObservableVariable'
            },
            selectionType: {
                keyName: 'selectionType',
                value: 'One'
            }
        },
        type: 'ButtonSetOne',
        extraProperties: (node: any) => generateToggleButtonProperties(node),
        connectedNodes: (node: any) => toggleConnectedNodes(node),
        customAttribute: {
            size: [
                {
                    value: "SM",
                    attrKey: "class",
                    attrValue: "'oj-button-sm'"
                },
                {
                    value: "LG",
                    attrKey: "class",
                    attrValue: "'oj-button-lg'"
                }
            ]
        }
    },
    sliderSingle: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            labelPosition: {
                keyName: "labelPosition"
            },
            min: {
                keyName: 'min',
                value: (_value: any, node: any) => extractMinMax(node, 'min')
            },
            max: {
                keyName: 'max',
                value: (_value: any, node: any) => extractMinMax(node, 'max')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            }
        },
        type: 'Slider'
    },
    checkboxSet: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            labelPosition: {
                keyName: "labelPosition"
            },
            orientation: {
                keyName: "orientation",
                value: (_value: any, node: any) => generateCheckBoxOrientation(node)
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            }
        },
        type: 'CheckBox',
        extraProperties: (node: any) => generateCheckBoxProperties(node),
    },
    button: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextForButton(node)
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            size: {
                keyName: 'size'
            },
            theme: {
                keyName: 'theme'
            },
            layoutSizingHorizontal: {
                keyName: 'layoutSizingHorizontal',
                value: 'FILL'
            },
            layoutSizingVertical: {
                keyName: 'layoutSizingVertical',
                value: 'HUG'
            }
        },
        type: 'Button',
        customAttribute: {
            state: [
                {
                    value: "Disabled",
                    attrKey: "disabled",
                    attrValue: "true"
                },
                {
                    value: "Read-only",
                    attrKey: "readonly",
                    attrValue: "true"
                }
            ],
            size: [
                {
                    value: "SM",
                    attrKey: "class",
                    attrValue: "'oj-button-sm'"
                },
                {
                    value: "LG",
                    attrKey: "class",
                    attrValue: "'oj-button-lg'"
                }
            ]
        },
        extraProperties: (node: any) => buttonIconProperties(node),

    },
    text: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            use: {
                keyName: 'use',
                value: 'ObservableVariable'
            },
            textBindingOption: {
                keyName: 'textBindingOption',
                value: 'NLS',
            },
            tagType: {
                keyName: 'tagType',
                value: (_value: any, node: any) => textTagTypeFunc(node)
            },
            customAttribute: {
                keyName: 'customAttribute',
                value: (_value: any, node: any) => generateTextCustomAttribute(node)
            }
        },
        type: 'Text',
    },
    'private/❗️text&Number': {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            use: {
                keyName: 'use',
                value: 'ObservableVariable'
            },
            textBindingOption: {
                keyName: 'textBindingOption',
                value: 'NLS',
            },
            tagType: {
                keyName: 'tagType',
                value: (_value: any, node: any) => {
                    return (node.parent.type === 'FRAME' && node.parent.layoutMode === 'HORIZONTAL')
                        ? 'Span'
                        : 'Div';
                }
            },
            customAttribute: {
                keyName: 'customAttribute',
                value: (_value: any, node: any) => generateTextCustomAttribute(node)
            }
        },
        type: 'Text',
    },
    link: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            hookFunction: {
                keyName: 'hookFunction',
                value: (_value: any, node: any) => addDrawer(node)
            },
            use: {
                keyName: 'use',
                value: 'ObservableVariable'
            },
            anchorType: {
                keyName: 'anchorType',
                value: 'Text',
            },
        },
        type: 'AnchorTag'
    },
    'link*': {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            use: {
                keyName: 'use',
                value: 'ObservableVariable'
            },
            anchorType: {
                keyName: 'anchorType',
                value: 'Text',
            },
        },
        type: 'AnchorTag',
    },
    meterBarVertical: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            percentage: {
                keyName: 'percentage',
            },
            nameholder: {
                keyName: 'nameholder',
                value: (() => `StatusGauge${generate()}`)()
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            orientationType: {
                keyName: 'orientationType',
                value: "'vertical'",
            },
            min: {
                keyName: 'min',
                value: 0,
            },
            max: {
                keyName: 'max',
                value: 100,
            },
        },
        type: 'StatusMeterGauge',
    },
    meterBarHorizontal: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            percentage: {
                keyName: 'percentage',
            },
            nameholder: {
                keyName: 'nameholder',
                value: (() => `StatusGauge${generate()}`)()
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            orientationType: {
                keyName: 'orientationType',
                value: "'horizontal'",
            },
            min: {
                keyName: 'min',
                value: 0,
            },
            max: {
                keyName: 'max',
                value: 100,
            },
        },
        type: 'StatusMeterGauge',
    },
    ratingGaugeStep: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (() => `RatingGauge${generate()}`)()
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            state: {
                keyName: 'state'
            }
        },
        type: 'RatingGauge',
        customAttribute: {
            size: [
                {
                    value: "LG",
                    attrKey: "size",
                    attrValue: "'lg'"
                },
                {
                    value: "MD",
                    attrKey: "size",
                    attrValue: "'md'"
                },
                {
                    value: "SM",
                    attrKey: "size",
                    attrValue: "'sm'"
                }
            ]
        }
    },
    ratingGaugeEditable: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (() => `RatingGauge${generate()}`)()
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            state: {
                keyName: 'state'
            }
        },
        type: 'RatingGauge',
        customAttribute: {
            size: [
                {
                    value: "LG",
                    attrKey: "size",
                    attrValue: "'lg'"
                },
                {
                    value: "MD",
                    attrKey: "size",
                    attrValue: "'md'"
                },
                {
                    value: "SM",
                    attrKey: "size",
                    attrValue: "'sm'"
                }
            ]
        }
    },
    avatarLight: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            label: {
                keyName: 'extraDetails',
                value: (_value: any, node: any) => `'${_value}'`
            },
            nameholder: {
                keyName: 'nameholder',
                value: (() => `Avatar${generate()}`)()
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            valueClasses: {
                keyName: 'valueClasses',
                value: "''"
            },
            size: {
                keyName: 'selectionType',
                value: (_value: any, node: any) => mapAvatarSize(_value)
            },
            type: {
                keyName: 'class'
            },
            icon: {
                keyName: 'iconClass',
                value: (_value: any, node: any) => fetchIconClass(_value, node)
            },
        },
        type: 'Avatar',
    },
    avatarDark: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            label: {
                keyName: 'extraDetails'
            },
            nameholder: {
                keyName: 'nameholder',
                value: (() => `Avatar${generate()}`)()
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            valueClasses: {
                keyName: 'valueClasses',
                value: "''"
            },
            size: {
                keyName: 'selectionType',
                value: (_value: any, node: any) => mapAvatarSize(_value)
            },
            type: {
                keyName: 'class'
            },
            icon: {
                keyName: 'iconClass',
                value: (_value: any, node: any) => fetchIconClass(_value, node)
            },
        },
        type: 'Avatar',
    },
    filePicker: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: 'File'
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            valueClasses: {
                keyName: 'valueClasses',
                value: "''"
            },
            children: {
                keyName: 'children',
                value: (node: any) => generateFileChildren(node)
            }
        },
        type: 'FilePicker',
    },
    badgeLight: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            tagType: {
                keyName: 'tagType',
                value: 'Span'
            },
            textBindingOption: {
                keyName: 'textBindingOption',
                value: 'NLS'
            }
        },
        type: 'Text',
        customAttribute: {
            semantic: [
                {
                    value: "Neutral",
                    attrKey: "class",
                    attrValue: "'oj-badge oj-badge-neutral'"
                },
                {
                    value: "Success",
                    attrKey: "class",
                    attrValue: "'oj-badge oj-badge-success'"
                },
                {
                    value: "Warning",
                    attrKey: "class",
                    attrValue: "'oj-badge oj-badge-warning'"
                },
                {
                    value: "Information",
                    attrKey: "class",
                    attrValue: "'oj-badge oj-badge-info'"
                },
                {
                    value: "Danger",
                    attrKey: "class",
                    attrValue: "'oj-badge oj-badge-danger'"
                }
            ]
        }
    },
    badgeDark: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => extractTextFromNode(node, 'TEXT')
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            tagType: {
                keyName: 'tagType',
                value: 'Span'
            },
            textBindingOption: {
                keyName: 'textBindingOption',
                value: 'NLS'
            }
        },
        type: 'Text',
        customAttribute: {
            semantic: [
                {
                    value: "Neutral",
                    attrKey: "class",
                    attrValue: "'oj-badge oj-badge-neutral'"
                },
                {
                    value: "Success",
                    attrKey: "class",
                    attrValue: "'oj-badge oj-badge-success'"
                },
                {
                    value: "Warning",
                    attrKey: "class",
                    attrValue: "'oj-badge oj-badge-warning'"
                },
                {
                    value: "Information",
                    attrKey: "class",
                    attrValue: "'oj-badge oj-badge-info'"
                },
                {
                    value: "Danger",
                    attrKey: "class",
                    attrValue: "'oj-badge oj-badge-danger'"
                }
            ]
        }
    },
    icon: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: (_value: any, node: any) => { return node?.children[0]?.name }
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            tagType: {
                keyName: 'tagType',
                value: 'Span'
            },
            customAttribute: {
                keyName: 'customAttribute',
                value: (_value: any, node: any) => {
                    return [{
                        'colon': true,
                        'attribute': 'class',
                        'value': `'oj-ux-ico-${node?.children[0]?.name}'`
                    }]
                }
            }
        },
        type: 'Text',
    },
    table: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            nameholder: {
                keyName: 'nameholder',
                value: `'Table${generate()}'`
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            use: {
                keyName: 'use',
                value: 'ObservableVariable'
            },
            source: {
                keyName: 'source',
                value: `dataSource${Math.floor(Math.random() * 10)}`
            },
            children: {
                keyName: 'children',
                value: (_value: any, node: any) => generateTableChildren(node)
            }
        },
        type: 'Table',
        extraProperties: (node: any) => generateTableProperties(node),
    },
    chart: {
        properties: {
            id: {
                keyName: 'id',
                value: generate()
            },
            mappedValue: {
                keyName: 'mappedValue',
                value: (_value: any, node: any) => generateMappedValue(node)
            },
            valueClasses: {
                keyName: 'valueClasses',
                value: "''"
            },
            chart: {
                keyName: 'chart',
                value: (_value: any, node: any) => fetchChartType(node)
            },
            slotTemplate: {
                keyName: 'slotTemplate',
                value: "<template slot='itemTemplate' data-oj-as='item'>\n  <oj-chart-item \n    value='[[item.data.value]]'\n    group-id='[[ [item.data.group] ]]'\n    series-id='[[item.data.series]]'> \n  </oj-chart-item>\n</template>"
            },
            customAttribute: {
                keyName: 'customAttribute',
                value: (_value: any, node: any) => chartCustomAttributes(node)
            },
            slots: {
                keyName: 'slots',
                value: (_value: any, node: any) => slotsFunc(node)
            },
            templateSlots: {
                keyName: 'templateSlots',
                value: (_value: any, node: any) => templateSlots(node)
            },
            children: {
                keyName: 'children',
                value: (_value: any, node: any) => slotChildren(node)
            }
        },
        type: "Chart",
    }
};
