import { getPropertiesDetails, getValueDetails, handleAddCondition, handleGrid, handleOverview, handleReadOnly, handleTransaction } from "./generateFunc";
import { metaData } from "./metadata";

export async function createClone(component: string, type: number) {

    if (component === 'pageSection' && type) {
        const componentSet = await figma.importComponentSetByKeyAsync("856c19a88057ba1893789e1b127c0d13f915f5d6") as ComponentSetNode;
        const componentNodeToClone = componentSet?.defaultVariant as ComponentNode;

        const frame = figma.createFrame();
        frame.name = `TransactionFrame`;

        frame.layoutMode = "VERTICAL";

        for (let i = 0; i < type; i++) {
            const clone = componentNodeToClone?.createInstance();
            clone && frame?.appendChild(clone);
        }

        frame.resize(componentNodeToClone?.width, type * componentNodeToClone?.height);

        frame.primaryAxisSizingMode = "AUTO";
        frame.counterAxisSizingMode = "AUTO";

        frame.x = (figma.viewport.center.x - frame.width) / 2;
        frame.y = (figma.viewport.center.y - frame.height) / 2;

        figma.currentPage.selection = [frame];
        figma.viewport.scrollAndZoomIntoView([frame]);

    } else if (component === 'overview' && type) {
        const componentNode = await figma.importComponentByKeyAsync("1208491d9a09690bbaa010b731fa6222139a8d19") as ComponentNode;
        const frame = figma.createFrame();
        frame.name = `OverviewFrame`;

        frame.layoutMode = "VERTICAL";

        for (let i = 0; i < type; i++) {
            const clone = componentNode?.createInstance();
            clone && frame?.appendChild(clone);
        }

        frame.resize(componentNode?.width, type * componentNode?.height);

        frame.primaryAxisSizingMode = "AUTO";
        frame.counterAxisSizingMode = "AUTO";

        frame.x = (figma.viewport.center.x - frame.width) / 2;
        frame.y = (figma.viewport.center.y - frame.height) / 2;

        figma.currentPage.selection = [frame];
        figma.viewport.scrollAndZoomIntoView([frame]);
    }
    figma.closePlugin();
}

export function generate() {
    figma.ui.resize(475, 450);
    function handleSelectionChange() {
        const selectedNodes = figma.currentPage.selection;
        //remove any
        let childrenList: any;
        let conditionData: any;
        let data: any;

        if (selectedNodes.length > 0) {
            const firstNode = selectedNodes[0];

            if ('children' in firstNode) {
                childrenList = firstNode.children.map((child: any) => child.name);
            }

            conditionData = firstNode.getPluginData("conditionData");
            const gridData = fetchGridData(firstNode);
            data = {
                selected: true,
                componentName: firstNode.name,
                componentType: firstNode.type,
                conditionData,
                fetchGridData: gridData,
                generateType: 'none'
            };
        } else {
            data = {
                selected: false,
                generateType: 'none'
            };
        }
        figma.ui.postMessage(data);
    }
    handleSelectionChange();
    figma.on('selectionchange', handleSelectionChange);
}

export async function generateOutput(selectedButton: string, messageDetails: any) {
    const selectedNodes = figma.currentPage.selection;
    if (selectedNodes?.length === 0) {
        figma.ui.postMessage({ selected: false });
        return;
    }

    const selectedComponent = selectedNodes[0];

    switch (selectedButton) {
        case 'readOnly':
            handleReadOnly(selectedComponent);
            break;

        case 'transaction':
            await handleTransaction(selectedComponent, messageDetails);
            break;

        case 'grid':
            handleGrid(selectedComponent);
            break;

        case 'condition':
            handleAddCondition(selectedComponent);
            break;

        default:
            figma.ui.postMessage({ type: 'error', message: 'Invalid button selected.' });
    }
}


export function conditionControlling() {
    figma.ui.hide();
    figma.once('selectionchange', () => {
        figma.ui.show();
        const selectedNodes = figma.currentPage.selection;
        if (selectedNodes.length === 0) {
            return;
        }
        const selectedComponent = selectedNodes[0] as InstanceNode;
        const mainComponent = selectedComponent.mainComponent as ComponentNode;
        const componentSetNode = mainComponent.parent as ComponentSetNode;
        const conditionComponent2 = {
            id: selectedComponent.id,
            name: selectedComponent.name,
            propertiesDetails: getPropertiesDetails(componentSetNode),
            componentValueDetails: getValueDetails(selectedComponent)
        };
        figma.ui.postMessage({ selected: true, conditionComponent2 });
    })
}

export function saveConditionData(conditionData: any) {
    const variableNode = figma.getNodeById(conditionData?.variableComponentId) as InstanceNode;
    const pluginConditionData = variableNode?.getPluginData("conditionData") || "[]";
    const pluginConditionArray = JSON.parse(pluginConditionData);
    pluginConditionArray.push(conditionData);
    const updatedJsonStr = JSON.stringify(pluginConditionArray);
    variableNode.setPluginData("conditionData", updatedJsonStr);

    figma.ui.resize(440, 450);
}

export function updateConditionData(conditionData: any) {
    const variableComponentId = Array.isArray(conditionData[0])
        ? conditionData[0][0].variableComponentId
        : conditionData[0].variableComponentId;

    const variableNode = figma.getNodeById(variableComponentId) as InstanceNode;

    if (variableNode) {
        const updatedJsonStr = JSON.stringify(conditionData);
        variableNode.setPluginData("conditionData", updatedJsonStr);
    } else {
        console.error(`Node with ID ${variableComponentId} not found.`);
    }
}

export function deleteCondition(index: any, figmaId: any) {
    const variableNode = figma.getNodeById(figmaId) as InstanceNode;
    const pluginConditionData = variableNode.getPluginData("conditionData");
    const pluginConditionArray = JSON.parse(pluginConditionData);
    const newConditionArray = pluginConditionArray.filter((_: any, indexno: any) => indexno !== index);
    const updatedJsonStr = JSON.stringify(newConditionArray);
    variableNode.setPluginData("conditionData", updatedJsonStr);

    const selectedNodes = figma.currentPage.selection;
    let data: any;

    if (selectedNodes.length > 0) {
        const firstNode = selectedNodes[0];

        data = {
            selected: true,
            componentName: firstNode.name,
            componentType: firstNode.type,
            conditionData: updatedJsonStr,
            generateType: 'none'
        };
    } else {
        data = {
            selected: false,
            generateType: 'none',
            conditionData: updatedJsonStr,
        };
    }
    figma.ui.postMessage(data);
}

export function saveGridData(gridData: any, selectedComponent: any) {
    const currentComponent = figma.getNodeById(selectedComponent.id) as InstanceNode;
    currentComponent?.setPluginData("flexExists", 'yes');
    for (let i = 0; i < gridData?.length; i++) {
        const childData = gridData[i];
        const child = currentComponent?.children[i];

        if (child) {
            child.setPluginData("flexWidth", JSON.stringify(childData));
        } else {
            console.warn(`Child at index ${i} not found in selected component`);
        }
    }
}

export function deleteGrid() {
    const selectedNodes = figma.currentPage.selection;
    let data: any;
    if (selectedNodes.length > 0) {
        const selectedComponent = selectedNodes[0] as InstanceNode;
        selectedComponent?.setPluginData("flexExists", '');

        for (let i = 0; i < selectedComponent?.children.length; i++) {
            const child = selectedComponent?.children[i];
            if (child) {
                child.setPluginData("flexWidth", '');
            }
        }

        data = {
            selected: true,
            componentName: selectedComponent?.name,
            componentType: selectedComponent?.type,
            fetchGridData: [],
            generateType: 'none'
        };
    } else {
        data = {
            selected: false,
            generateType: 'none',
            fetchGridData: []
        };
    }
    figma.ui.postMessage(data);
}

export function fetchGridData(selectedComponent: any): any[] {
    const gridData: any[] = [];

    const flexExists = selectedComponent.getPluginData("flexExists");
    if (flexExists === 'yes') {
        for (let i = 0; i < selectedComponent.children.length; i++) {
            const child = selectedComponent.children[i];
            const flexWidthData = child.getPluginData("flexWidth");

            if (flexWidthData) {
                try {
                    const parsedData = JSON.parse(flexWidthData);
                    gridData.push(parsedData);
                } catch (e) {
                    console.error(`Error parsing flexWidth data for child at index ${i}:`, e);
                }
            } else {
                console.warn(`No flexWidth data found for child at index ${i}`);
            }
        }
    } else {
        console.warn("No flex data exists for the selected component.");
    }

    return gridData;
}

let selectionChangeListener: (() => void) | undefined;

export function removeSelectionChangeListener() {
    if (selectionChangeListener) {
        figma.off('selectionchange', selectionChangeListener);
        selectionChangeListener = undefined;
    }
}

export function selectTemplateComponent() {
    figma.ui.hide();
    figma.once('selectionchange', () => {
        figma.ui.show();
        const selectedNodes = figma.currentPage.selection;
        if (selectedNodes.length > 0) {
            const selectedComponent = selectedNodes[0];
            const templateData = {
                id: selectedComponent?.id,
                name: selectedComponent?.name,
                nodeType: selectedComponent?.type
            }
    
            figma.ui.postMessage({ selected: true, templateComponent: templateData });
        }
    })
}

export function selectOverviewEmptyComponent() {
    figma.ui.hide();
    figma.once('selectionchange', () => {
        figma.ui.show();
        const selectedNodes = figma.currentPage.selection;
        if (selectedNodes.length > 0) {
            const selectedComponent = selectedNodes[0];
            const templateData = {
                id: selectedComponent?.id,
                name: selectedComponent?.name,
                nodeType: selectedComponent?.type
            }
    
            figma.ui.postMessage({ selected: true, overviewNoDataComponent: templateData });
        }
    })
}

export function selectOverviewSidePanelComponent() {
    figma.ui.hide();
    figma.once('selectionchange', () => {
        figma.ui.show();
        const selectedNodes = figma.currentPage.selection;
        if (selectedNodes.length > 0) {
            const selectedComponent = selectedNodes[0];
            const templateData = {
                id: selectedComponent?.id,
                name: selectedComponent?.name,
                nodeType: selectedComponent?.type
            }
            figma.ui.postMessage({ selected: true, overviewSidePanel: templateData });
        }

    })
}

export function templateComponent() {
    const selectedNodes = figma.currentPage.selection;
    if (selectedNodes.length === 0) {
        return;
    }
    const selectedComponent = selectedNodes[0];
    const templateComponent = {
        id: selectedComponent.id,
        name: selectedComponent.name,
        nodeType: selectedComponent.type
    }
    figma.ui.postMessage({ templateComponent: templateComponent });

}

export async function templateOutput(selectedButton: string, messageDetails: any) {

    switch (selectedButton) {

        case 'transaction':
            const selectedComponent = figma.currentPage.findOne((n: any) => n.id === messageDetails?.id)
            await handleTransaction(selectedComponent, messageDetails?.type);
            break;

        case 'overview':
            await handleOverview(messageDetails);
            break;

        default:
            figma.ui.postMessage({ type: 'error', message: 'Invalid button selected.' });
    }
}

export function exportResult(formData: any) {
    selectionChangeListener = () => processSelection(formData);
    figma.on('selectionchange', selectionChangeListener);
}

export const processSelection = async (formData: any) => {
    const selectedNodes = figma.currentPage.selection;

    if (selectedNodes?.length === 0) {
        figma.ui.postMessage({ selected: false });
        figma.ui.postMessage({ clearIgnoredList: true });
    } else {
        figma.ui.postMessage({ clearIgnoredList: true });
        const componentName = selectedNodes[0]?.name;
        figma.ui.postMessage({ clearIgnoredList: false });
        const output: any = await metaData(selectedNodes[0], formData);
        if (output) {
            figma.ui.postMessage({ output });
        } else {
            console.log('Error in Output');
        }

        figma.ui.postMessage({
            // clearIgnoredList: true,
            selected: true,
            componentName,
            output,
        });
    }
};
