import { useState, useContext, useEffect } from "preact/hooks";
import { createContext } from "preact";

type PluginMessageContent = {
    path?: string;
    selected?: boolean;
    action?: string;
    componentName?: string;
    componentType?: string;
    conditionComponent1?: any;
    conditionComponent2?: any;
    conditionData?: any;
    generateType?: string;
    gridComponent?: any;
    selectedGridComponent?: any;
    fetchGridData?: any;
    templateComponent?: any;
    overviewNoDataComponent?: any;
    overviewSidePanel?: any;
    output?: any;
    drawerMetaData?: any;
    drawerFormData?: any;
    pluginSelectedComponent?: string;
    error?: string;
    ignoredInstance?: { name: string; id: string; knownComponentExists: boolean };
    clearIgnoredList?: boolean
};

type PluginMessage = {
    pluginMessage: PluginMessageContent;
    pluginId: string;
};

type PluginDataContextType = {
    pluginData: PluginMessageContent | undefined;
    setPluginData: (type: PluginMessageContent) => void;
    ignoredComponentListKnown: IgnoredComponent[];
    ignoredComponentListUnknown: IgnoredComponent[];
};

type IgnoredComponent = { name: string; id: string };

const AppContext = createContext<PluginDataContextType | undefined>(undefined);

const AppDataProvider = ({ children }: any) => {

    const [pluginData, setPluginData] = useState<PluginMessageContent | undefined>(undefined);

    const [ignoredComponentListKnown, setIgnoredComponentListKnown] = useState<IgnoredComponent[]>([]);
    const [ignoredComponentListUnknown, setIgnoredComponentListUnknown] = useState<IgnoredComponent[]>([]);

    useEffect(() => {
        const handleMessage = (event: MessageEvent<PluginMessage>) => {
            const pluginMessage = event.data.pluginMessage;
            const { action, ignoredInstance, clearIgnoredList } = pluginMessage;
            setPluginData(prevPluginData => ({
                ...prevPluginData,
                ...pluginMessage,
                selected: pluginMessage.selected ?? false
            }));
            if (ignoredInstance) {
                const { name, id, knownComponentExists } = ignoredInstance;
                if (knownComponentExists) {
                    setIgnoredComponentListKnown((prevList: IgnoredComponent[]) => [
                        ...prevList,
                        { name, id }
                    ]);
                } else {
                    setIgnoredComponentListUnknown((prevList: IgnoredComponent[]) => [
                        ...prevList,
                        { name, id }
                    ]);
                }
            }
            

            if (clearIgnoredList) {
                // setFormData({
                //     ...formData,
                //     ignoredInstanceList: []
                // });
                setIgnoredComponentListKnown([]);
                setIgnoredComponentListUnknown([]);
            }

            if (action === 'close') {
                parent.postMessage({ pluginMessage: { type: 'closePlugin' } }, '*');
            }
        };

        window.addEventListener('message', handleMessage);

        return () => {
            window.removeEventListener('message', handleMessage);
        };
    }, []);

    return (
        <AppContext.Provider 
        value={{
            pluginData,
            setPluginData,
            ignoredComponentListKnown,
            ignoredComponentListUnknown,
        }}
        >
            {children}
        </AppContext.Provider>
    );
};

const useAppDataContext = (): PluginDataContextType => {
    const context = useContext(AppContext);
    if (!context) {
        throw new Error('useAppDataContext must be used within an AppDataProvider');
    }
    return context;
};

export { AppDataProvider, useAppDataContext };
