import { camelCase, camelToHyphen, generate } from "./cleanFunc";
import { extractTextFromNode } from "./components/commonFunc";
import { slotToTagType } from "./maps/componentMap";
import { setIgnoreKey } from "./pluginData";
import { formDataFunc } from "./processFunc";
import { getListViewGridLinesStatus } from "./toolkitFunc";
import { traverseNode } from "./traverseNode";

export function getComponentJson(componentList: any, componentName: string) {
    const name = camelToHyphen(componentName);
    const componentJson = componentList && componentList[0] && componentList.find((comp: any) => comp.name === name);
    if (componentJson) {
        return componentJson;
    } else {
        return false;
    }
}

// export function toolkitCleanProperties(properties: any): any {
//     try {
//         const output: any = {};

//         const toCamelCase = (str: string): string => {
//             return str
//                 .replace(/(?:^\w|[A-Z]|\b\w|\s+|\W)/g, (match, index) =>
//                     index === 0 ? match.toLowerCase() : match.toUpperCase()
//                 )
//                 .replace(/\s+/g, '');
//         };

//         Object.keys(properties).forEach((key) => {
//             const cleanKey = toCamelCase(key.split("#")[0]);
//             output[cleanKey] = properties[key];
//         });

//         return output;
//     } catch (error) {
//         console.error("Error in toolkitCleanProperties:", error);
//         return {};
//     }
// }

export function toolkitCleanProperties(properties: any): any {
    try {
        const output: any = {};

        // const toCamelCase = (str: string): string => {
        //     return str
        //         .replace(/[^a-zA-Z0-9\s]/g, '')
        //         .replace(/(?:^\w|[A-Z]|\b\w|\s+)/g, (match, index) =>
        //             index === 0 ? match.toLowerCase() : match.toUpperCase()
        //         )
        //         .replace(/\s+/g, '');
        // };

        Object.keys(properties).forEach((key) => {
            const cleanKey = camelCase(key.split("#")[0]);
            output[cleanKey] = properties[key];
        });

        return output;
    } catch (error) {
        console.error("Error in toolkitCleanProperties:", error);
        return {};
    }
}


export function frameWorkComponents(name: any, component: any, node: any) {
    interface Result {
        nameholder: string;
        componentName: any;
        params: any;
        type: string;
        componentID: string;
        valueClasses: string;
        figmaId: string;
        slots?: { slotName: string }[];
        children?: any[];
        templateSlots?: { slotName: string }[];
    }

    try {
        const formComponentName: string = formDataFunc().componentName;

        const componentInfo: any = {
            widget: {
                componentName: "obdx-widget",
                fixedParams: [
                    {
                        "name": "id",
                        "type": "{valueType}"
                    }
                ]
            },
            modalDialog: {
                componentName: "modal-window",
                fixedParams: [
                    {
                        "name": "id",
                        "type": "{valueType}"
                    },
                    {
                        "name": "dialog-id",
                        "value": formComponentName ? `'${formComponentName}-${generate()}'` : `'moduleWindow-${generate()}'`
                    }
                ]
            },
            amountInput: {
                componentName: "amount-input",
                fixedParams: [
                    {
                        "name": "id",
                        "type": "{valueType}"
                    },
                    {
                        "name": "root-id",
                        "value": `'root-${generate()}'`
                    }
                ]
            },
            account: {
                componentName: "account-input",
                fixedParams: []
            },
            row: {
                componentName: "obdx-row",
                fixedParams: []
            },
            internalAccountInput: {
                componentName: "internal-account-input",
                fixedParams: []
            },
            address: {
                componentName: "obdx-address",
                fixedParams: []
            },
            addressInput: {
                componentName: "address-input",
                fixedParams: []
            },
            uiDownload: {
                componentName: "transaction-container",
                fixedParams: []
            },
            quickActionButton: {
                componentName: "quick-action-button",
                fixedParams: []
            },
            obdxLookup: {
                componentName: "bank-look-up",
                fixedParams: []
            },
            obdxActionCard: {
                componentName: "obdx-action-card",
                fixedParams: [{
                    "name": "avatar",
                    "value": "{initials: 'U X'}",
                    "type": "{valueType}"
                }]
            },
            obdxWidget: {
                componentName: "obdx-widget",
                fixedParams: []
            }
            // Add information for other components as needed
        };

        const mappingRules: Record<string, any> = {
            quickActionButton: {
                label: {
                    key: 'primaryText'
                },
                size: {
                    key: 'size',
                    values: {
                        '196px/2XL': 'xxl',
                        '144px/XL': 'xl',
                        '96px/LG': 'lg',
                        '72px/MD': 'md',
                        '56px/SM': 'sm',
                        '44px/XS': 'xs',
                        '28px/2XS': 'xxs'
                    }
                },
                initials: {
                    key: 'label'
                },
                icon: {
                    key: 'icon',
                    customValue: (value: string) => `oj-ux-ico-${figma.getNodeById(value)?.parent?.name}`
                },
                color: {
                    key: 'color',
                    customValue: (value: string) => `${(figma.getNodeById(value)?.name)?.toLowerCase()}`
                },
                shape: {
                    key: 'shape',
                    customValue: (value: string) => `${value.toLowerCase()}`
                },
                'custom-slot': {
                    key: 'customSlot'
                },
                action: {
                    key: 'action',
                    customValue: 'quick1'
                }

            },
            address: {
                accountaddressenable: {
                    key: 'addressEnabled'
                }
            },
            modalWindow: {
                Body: {
                    key: 'body'
                },
                Footer: {
                    key: 'footer'
                },
            },
            obdxActionCard: {
                badgeText: {
                    key: 'text'
                },
                badgeType: {
                    key: 'semantic',
                    customValue: (value: string) => `${value.toLowerCase()}`
                }
            }
        };

        const getMappedKey = (component: string, key: string) => mappingRules[component]?.[key]?.key || key;


        const getMappedValue = (component: string, key: string, value: string) =>
            mappingRules[component]?.[key]?.values?.[value] || value;

        const getFinalMappedValue = (component: string, key: string, value: string) => {
            const mappingRule = mappingRules[component]?.[key];
            if (mappingRule?.values) {
                return getMappedValue(component, key, value);
            }
            if (mappingRule?.customValue) {
                return getCustomMappedValue(component, key, value);
            }
            return value;
        };

        const getCustomMappedValue = (component: string, key: string, value: string) => {
            const mappingRule = mappingRules[component]?.[key];
            if (mappingRule?.customValue) {
                return typeof mappingRule.customValue === 'function' ? mappingRule.customValue(value) : value;
            }
            return mappingRule?.values?.[value] || value;
        };

        const componentData: any = componentInfo[name];
        if (!componentData) {
            throw new Error(`Unsupported component type: ${name}`);
        }

        const result: Result = {
            nameholder: `'${generate()}'`,
            componentName: componentData.componentName,
            params: componentData?.fixedParams,
            type: "FrameworkElement",
            componentID: `'${generate()}'`,
            valueClasses: "''",
            figmaId: node.id
        };

        const combinedProperties = {
            ...node.componentProperties,
            ...Object.assign({}, ...node.exposedInstances.map((instance: any) => instance.componentProperties))
        };
        let output = toolkitCleanProperties(combinedProperties);
        for (const [key, value] of Object.entries(component.properties as Record<string, any>)) {
            const mappedKey: string = getMappedKey(name, key);
            if (value.type.includes('string')) {
                let paramsResult: ParamsResult = {};
                type ParamsResult = {
                    value?: string;
                    name?: string;
                    type?: string;
                    translatable?: boolean
                }
                const foundNode = node.findOne((n: any) => n.name === key);
                if (foundNode !== null) {
                    const label = extractTextFromNode(foundNode, 'TEXT');
                    paramsResult = {
                        name: camelToHyphen(key),
                        value: `'${label}'`,
                        type: "{valueType}",
                    }
                    if ((value as { translatable?: boolean }).translatable) {
                        paramsResult.translatable = true;
                    }
                    result.params.push(paramsResult);
                } else if (mappedKey in output) {
                    const mappedValue: string = getFinalMappedValue(name, key, output[mappedKey].value);
                    paramsResult = {
                        name: camelToHyphen(key),
                        value: `'${mappedValue}'`,
                        type: "{valueType}",
                    }
                    if ((value as { translatable?: boolean }).translatable) {
                        paramsResult.translatable = true;
                    }
                    result.params.push(paramsResult);
                }
            }
            else if (value.type === 'boolean') {
                if (mappedKey in output) {
                    const paramsResult = {
                        name: camelToHyphen(key),
                        value: String(output[mappedKey].value).toLowerCase(),
                        type: "{valueType}",
                    }
                    result.params.push(paramsResult);
                }
            }
        }
        const slots: { slotName: string }[] = [];
        const templateSlots: { slotName: string }[] = [];
        const children: any = [];
        if (component?.slots) {
            for (let [key, value] of Object.entries(component?.slots)) {
                const mappedKey: string = getMappedKey(name, key);
                if (key === '') {
                    key = 'Default';
                }
                slots.push({
                    slotName: key
                });
                if (mappedKey in output) {

                    templateSlots.push({
                        slotName: key
                    });

                    const obj = node.findOne((n: any) => {
                        const cleanedName = camelCase(n.name);
                        // return (cleanedName[0] === mappedKey);
                        return (cleanedName === mappedKey);
                    })?.children;

                    const traverseResult = obj?.map((element: any) => traverseNode(element)) || [];
                    const slotChildren = Array.isArray(traverseResult) ? traverseResult.flat() : traverseResult;

                    if (slotChildren.length > 0 && slotChildren.some((child: any) => !(typeof child === 'object' && !Array.isArray(child) && Object.keys(child).length === 0))) {
                        children.push({
                            nameholder: key,
                            subType: "slot",
                            containerType: "NormalContainer",
                            valueClasses: "''",
                            type: "Container",
                            componentID: `'${generate()}'`,
                            children: slotChildren
                        })
                    }
                }
            }
            if (children.length > 0) {
                if (templateSlots?.length > 0) {
                    result.templateSlots = templateSlots;
                }
                result.children = children;
            }
            if (slots?.length > 0)
                result.slots = slots;
        }
        return result;
    } catch (error) {
        console.error("Error in frameWorkComponents:", error);
        throw new Error("An unexpected error occurred");
    }
}

export function businessComponents(component: any, node: any) {

    interface Result {
        nameholder?: string;
        loaderType?: string;
        moduleName?: string;
        componentName?: string;
        params?: any;
        type?: string;
        componentID?: string;
        figmaId: string
    }

    try {
        const componentName = component?.name;
        const componentDescription = componentName?.split('-')
            .map((word: string) => word.charAt(0).toUpperCase() + word.slice(1))
            .join(' ');

        const moduleName = formDataFunc().moduleName || component.moduleName;
        const result: Result = {
            nameholder: componentDescription,
            loaderType: 'Component',
            moduleName: moduleName,
            componentName: componentName,
            type: 'Business',
            componentID: `'${generate()}'`,
            figmaId: node.id,
            params: []
        }
        let output = toolkitCleanProperties(node?.componentProperties);
        for (const [key, value] of Object.entries(component.properties as Record<string, any>)) {
            if (value.type.includes('string')) {
                let paramsResult: ParamsResult = {};
                type ParamsResult = {
                    value?: string;
                    name?: string;
                    type?: string;
                    translatable?: boolean
                }
                const foundNode = node.findOne((n: any) => n.name === key);
                if (foundNode !== null) {
                    const label = extractTextFromNode(foundNode, 'TEXT');
                    paramsResult = {
                        name: key,
                        value: `'${label}'`,
                        type: "{valueType}",
                    }
                    if ((value as { translatable?: boolean }).translatable) {
                        paramsResult.translatable = true;
                    }
                    result.params.push(paramsResult);
                } else if (key in output) {
                    paramsResult = {
                        name: key,
                        value: `'${output[key].value.replace(/ /g, "").toUpperCase()}'`,
                        type: "{valueType}",
                    }
                    if ((value as { translatable?: boolean }).translatable) {
                        paramsResult.translatable = true;
                    }

                    result.params.push(paramsResult);
                }
            }
            else if (value.type === 'boolean') {
                if (key in output) {
                    const paramsResult = {
                        name: key,
                        value: String(output[key].value).toLowerCase(),
                        type: "{valueType}",
                    }
                    result.params.push(paramsResult);
                } else if (key === 'readonly') {
                    result.params.push({
                        name: 'readonly',
                        value: "false",
                    })
                }
            }
        }
        return result;
    } catch (error) {
        console.error("Error in businessWorkComponents:", error);
        throw new Error("An unexpected error occurred");
    }
}

export function coreComponents(component: any, node: any) {
    interface Result {
        componentName: any;
        params?: any;
        type: string;
        componentID: string;
        figmaId: string;
        slots?: { name: string }[];
        children?: any[];
        templateSlots?: { slotName: string }[];
    }

    try {

        const componentInfo: any = {
            "list-view": {
                componentName: "list-view",
                fixedParams: [
                    {
                        "name": "id",
                        "type": "{valueType}"
                    },
                    {
                        "displayName": "Selection Mode",
                        "value": "'none'",
                        "type": "string",
                        "name": "selectionMode"
                    },
                    {
                        "displayName": "Data",
                        "value": "dataProvider",
                        "type": "DataProvider|null",
                        "name": "data"
                    },
                ]
            },
            "legend-item": {
                componentName: "legend-item",
                fixedParams: [
                    {
                        "displayName": "Symbol Type",
                        "value": "'marker'",
                        "type": "string",
                        "name": "symbolType"
                    },
                    {
                        "displayName": "Line Style",
                        "value": "'solid'",
                        "type": "string",
                        "name": "lineStyle"
                    },
                    {
                        "displayName": "Marker Shape",
                        "value": "'square'",
                        "type": "string",
                        "name": "markerShape"
                    },
                    {
                        "displayName": "Drilling",
                        "value": "'inherit'",
                        "type": "string",
                        "name": "drilling"
                    },
                ]
            },
            legend: {
                componentName: "legend",
                fixedParams: [
                    {
                        "displayName": "Data",
                        "value": "dataProvider",
                        "type": "DataProvider|null",
                        "name": "data"
                    },
                    {
                        "displayName": "Drilling",
                        "value": "'off'",
                        "type": "string",
                        "name": "drilling"
                    },
                    {
                        "displayName": "Halign",
                        "value": "'center'",
                        "type": "string",
                        "name": "halign"
                    },
                    {
                        "displayName": "Hide And Show Behavior",
                        "value": "'off'",
                        "type": "string",
                        "name": "hideAndShowBehavior"
                    },
                    {
                        "displayName": "Hover Behavior",
                        "value": "'none'",
                        "type": "string",
                        "name": "hoverBehavior"
                    },
                    {
                        "displayName": "valign",
                        "value": "'middle'",
                        "type": "string",
                        "name": "valign"
                    },
                    {
                        "displayName": "Section Title Halign",
                        "value": "'center'",
                        "type": "string",
                        "name": "sectionTitleHalign"
                    },
                ]
            }
        }
        const componentData: any = componentInfo[component?.name];


        const mappingRules: Record<string, any> = {
            collapsible: {
                expanded: {
                    key: 'expanded',
                    values: {
                        'Yes': true,
                        'No': false
                    }
                },
                iconPosition: {
                    key: 'position',
                    values: {
                        'Leading': 'start',
                        'Trailing': 'end'
                    }
                }
            },
            "list-view": {
                gridlines: {
                    key: 'gridlines',
                    customValue: (value: string) => getListViewGridLinesStatus(value)
                },
                selectionMode: {
                    key: 'selectionMode',
                    customValue: () => 'none'
                }
            }
        };

        const getMappedKey = (component: string, key: string) => mappingRules[component]?.[key]?.key || key;


        const getMappedValue = (component: string, key: string, value: string) =>
            mappingRules[component]?.[key]?.values?.[value];

        const getFinalMappedValue = (component: string, key: string, value: string) => {

            const mappingRule = mappingRules[component]?.[key];
            if (mappingRule?.values) {
                return getMappedValue(component, key, value);
            }
            if (mappingRule?.customValue) {
                return getCustomMappedValue(component, key, value);
            }
            return value;
        };

        const getCustomMappedValue = (component: string, key: string, value: string) => {
            const mappingRule = mappingRules[component]?.[key];
            // if (mappingRule?.customValue) {
            return typeof mappingRule.customValue === 'function' ? mappingRule.customValue(value) : value;
            // }
            // return mappingRule?.values?.[value] || value;
        };

        const result: Result = {
            componentName: component.name,
            type: "CoreComponent",
            componentID: `'${generate()}'`,
            figmaId: node.id,
            params: componentData?.fixedParams || []
        };

        const combinedProperties = {
            ...node.componentProperties,
            ...Object.assign({}, ...node.exposedInstances.map((instance: any) => instance.componentProperties))
        };

        let output = toolkitCleanProperties(combinedProperties);
        if (component.properties) {
            for (const [key, value] of Object.entries(component.properties as Record<string, any>)) {
                const mappedKey: string = getMappedKey(component.name, key);
                if (mappedKey in output) {
                    const mappedValue: any = getFinalMappedValue(component.name, key, output[mappedKey].value);
                    const formattedValue = value?.type === 'boolean' || value?.type === 'object' ? mappedValue : `'${mappedValue}'`;
                    result.params.push({
                        displayName: value?.displayName || key,
                        value: formattedValue,
                        type: value?.type,
                        name: key
                    });
                }
            }
        }

        if (component?.slots) {
            const slots: { name: string }[] = [];
            const templateSlots: { slotName: string, slotTag: string }[] = [];
            const children: any = [];
            for (let [key, value] of Object.entries(component?.slots)) {
                if (key === '') {
                    key = 'defaultSlot';
                }
                const slotKey = key === 'defaultSlot' ? 'default' : key;
                slots.push({
                    name: slotKey
                });
                if (key in output) {
                    const obj = node.findOne((n: any) => {
                        const cleanedName = camelCase(n.name);
                        return cleanedName === key && n.visible;
                    })?.children;
                    const traverseResult = obj?.map((element: any) => traverseNode(element)).filter((result: any) => Object.keys(result).length > 0) || [];
                    const slotChildren = Array.isArray(traverseResult) ? traverseResult.flat() : traverseResult;
                    const slotTag = slotToTagType(key);
                    if (slotChildren.length > 0) {
                        templateSlots.push({
                            slotName: slotKey,
                            slotTag: slotTag
                        });
                        children.push({
                            nameholder: slotKey,
                            subType: "slot",
                            slotTag: slotTag,
                            containerType: "NormalContainer",
                            valueClasses: "''",
                            type: "Container",
                            componentID: `'${generate()}'`,
                            children: slotChildren.flat()
                        })
                    }
                }
            }
            if (templateSlots.length) result.templateSlots = templateSlots;
            if (slots.length) result.slots = slots;
            if (children.length) result.children = children;
        }
        return result;
    } catch (error) {
        console.error("Error in coreComponents:", error);
        throw new Error("An unexpected error occurred");
    }
}

export function spectraComponent(component: any, node: any) {
    interface Result {
        nameholder: string;
        componentName: string;
        params?: any;
        valueClasses: string;
        type: string;
        componentID: string;
        figmaId: string
    }

    const propertyMapper: any = {
        card: {
            overlineText: { figmaKey: 'Overline' },
            primaryText: { figmaKey: 'Title' },
            secondaryText: { figmaKey: 'Secondary Text' },
            tertiaryText: { figmaKey: 'Tertiary Text' },
            imageSize: { figmaKey: 'image' },
            badge: {
                figmaKey: 'Badge - Light',
                subObjectMap: {
                    status: 'Semantic',
                    style: 'Theme'
                }
            }
        }
    }

    try {
        const result: Result = {
            nameholder: `'${generate()}'`,
            componentName: component?.name,
            valueClasses: "''",
            type: 'SpectraComponent',
            componentID: `'${generate()}'`,
            params: [{
                "name": "id",
                "type": "{valueType}",
                "value": `'${component?.name}${generate()}'`
            }],
            figmaId: node.id

        }
        let output = toolkitCleanProperties(node?.componentProperties);
        for (const [key, value] of Object.entries(component.properties as Record<string, any>)) {
            const componentName = component?.name;
            const componentMapProperties = propertyMapper[componentName];
            let finalKey: string;
            if (componentMapProperties && componentMapProperties[key]) {
                finalKey = componentMapProperties[key].figmaKey;
            } else {
                finalKey = key;
            }
            if (value.type.includes('object') && value.properties) {
                const objectNode = node.findOne((n: any) => n.name === finalKey);
                if (objectNode) {
                    const valueProperties: any = value.properties;
                    let output = toolkitCleanProperties(objectNode.componentProperties);
                    let objResult: any = {};
                    for (const [subKey, subValue] of Object.entries(valueProperties as Record<string, any>)) {
                        let subObjectKey: string;
                        const subObjectMap: any = componentMapProperties[key].subObjectMap;
                        if (subObjectMap && subObjectMap[subKey]) {
                            subObjectKey = subObjectMap[subKey].toLowerCase();
                        } else {
                            subObjectKey = subKey;
                        }
                        if (subObjectKey in output && subValue.enumValues) {
                            objResult[subKey] = (output[subObjectKey].value)?.toLowerCase();
                        } else if (subObjectKey in output) {
                            objResult[subKey] = (output[subObjectKey].value);
                        }
                    }
                    const objParamsResult = {
                        name: camelToHyphen(key),
                        type: "{valueType}",
                        value: objResult
                    }
                    result.params.push(objParamsResult);
                }
            }
            if (value.type.includes('string')) {
                let paramsResult: ParamsResult = {};
                type ParamsResult = {
                    value?: string;
                    name?: string;
                    type?: string;
                    translatable?: boolean
                }
                const foundNode = node.findOne((n: any) => n.name === finalKey);
                if (finalKey in output) {
                    let value = (output[finalKey].value)?.toLowerCase();
                    paramsResult = {
                        name: camelToHyphen(key),
                        value: value === '-' ? `'none'` : `'${value}'`,
                        type: "{valueType}",
                    }
                    if ((value as { translatable?: boolean }).translatable) {
                        paramsResult.translatable = true;
                    }
                    result.params.push(paramsResult);
                } else if (foundNode !== null) {
                    const label = extractTextFromNode(foundNode, 'TEXT');
                    if (label !== '') {
                        paramsResult = {
                            name: camelToHyphen(key),
                            value: `'${label}'`,
                            type: "{valueType}",
                        }
                        if ((value as { translatable?: boolean }).translatable) {
                            paramsResult.translatable = true;
                        }
                        result.params.push(paramsResult);
                    }
                }
            } else if (value.type === 'boolean') {
                if (finalKey in output) {
                    const paramsResult = {
                        name: camelToHyphen(key),
                        value: String(output[finalKey].value).toLowerCase(),
                        type: "{valueType}",
                    }
                    result.params.push(paramsResult);
                }
            }
        }
        return result;
    } catch (error) {
        console.error("Error in spectraWorkComponents:", error);
        throw new Error("An unexpected error occurred");
    }
}

let modals: any = [];

export function clearModals() {
    modals = [];
}

export function buttonModalReaction(node: any) {
    try {
        if (!node) {
            throw new Error("The 'node' parameter is missing or undefined. in extract Text from Node");
        }
        let destId = '';
        for (let i of node.reactions) {
            if (i.trigger.type === 'ON_CLICK') {
                destId = i.action.destinationId;
            }
        }
        const destNode: any = figma.getNodeById(destId);
        if (destNode?.name === 'Modal Dialog') {
            modals.push(traverseNode(destNode));
            setIgnoreKey(destNode.id);
        }
    }
    catch (error) {
        console.error(error);
        return '';
    }
}
export function modalWindows() {
    return modals;
}