##################################################################################
# OKE Advanced Terraform Root Module                                             #
# File: ~/oke_advanced_module/main.tf                                            #
#                                                                                #
# Description:                                                                   #
#   Root orchestration for Oracle Kubernetes Engine (OKE) advanced deployments.  #
#   Provisions VCN, cluster, node pools, and bastion based on provided           #
#   variable values, using reusable modules and best practices.                  #
#                                                                                #
# Authors: Mahamat H. Guiagoussou, Payal Sharma, Matthew McDaniel                #
# Copyright (c) 2025 Oracle                                                      #
##################################################################################


# Root Module Data Sources
data "oci_identity_availability_domains" "availability_domains" {
  # Tenancy OCID is the root compartment.
  compartment_id = var.tenancy_ocid
}


#--------------------------------------------------------------------------------
# STEP 1: Create the VCN and Related Network Resources                           #
#--------------------------------------------------------------------------------
module "vcn" {
  count                                         = var.is_vcn_created ? 1 : 0
  source                                        = "./modules/vcn"
  compartment_id                                = var.is_networking_compartment_separate ? var.networking_compartment_id : var.compartment_id
  vcn_cidr_block                                = var.vcn_cidr_block
  display_name_prefix                           = "${var.display_name_prefix}-1"
  host_name_prefix                              = var.host_name_prefix
  k8apiendpoint_private_subnet_cidr_block       = var.k8apiendpoint_private_subnet_cidr_block
  workernodes_private_subnet_cidr_block         = var.workernodes_private_subnet_cidr_block
  pods_private_subnet_cidr_block                = var.pods_private_subnet_cidr_block
  serviceloadbalancers_public_subnet_cidr_block = var.serviceloadbalancers_public_subnet_cidr_block
  bastion_public_subnet_cidr_block              = var.bastion_public_subnet_cidr_block
  use_nsg                                       = var.use_nsg
}


#--------------------------------------------------------------------------------
# STEP 2: Create the OKE Cluster                                                 #
#--------------------------------------------------------------------------------
module "oke" {
  depends_on                            = [module.vcn[0]]
  count                                 = var.is_k8cluster_created && var.is_vcn_created ? 1 : 0
  source                                = "./modules/oke"
  compartment_id                        = var.compartment_id
  vcn_id                                = module.vcn[0].vcn_id
  k8apiendpoint_private_subnet_id       = module.vcn[0].k8apiendpoint_private_subnet_id
  workernodes_private_subnet_id         = module.vcn[0].workernodes_private_subnet_id
  pods_private_subnet_id                = module.vcn[0].nodepods_private_subnet_id
  serviceloadbalancers_public_subnet_id = module.vcn[0].serviceloadbalancers_public_subnet_id

  # Cluster configuration
  cluster_freeform_tag_key         = var.cluster_freeform_tag_key
  cluster_freeform_tag_value       = var.cluster_freeform_tag_value
  cluster_name                     = "${var.display_name_prefix}-Cluster"
  control_plane_kubernetes_version = var.control_plane_kubernetes_version
  control_plane_is_public          = var.control_plane_is_public
  cni_type                         = var.cni_type
  is_cni_type_native               = (var.cni_type == "OCI_VCN_IP_NATIVE")

  # Worker Pools
  node_pool_freeform_tag_key      = var.node_pool_freeform_tag_key
  node_pool_freeform_tag_value    = var.node_pool_freeform_tag_value
  image_signing_enabled           = var.image_signing_enabled
  image_signing_key_id            = var.image_signing_key_id
  cluster_type                    = var.cluster_type
  worker_nodes_kubernetes_version = var.worker_nodes_kubernetes_version
  is_node_pools_created           = var.is_nodepool_created
  node_pools                      = var.worker_node_pools

  use_nsg             = var.use_nsg
  k8api_nsg_id        = var.use_nsg ? module.vcn[0].k8api_nsg_id : null
  worker_nodes_nsg_id = var.use_nsg ? module.vcn[0].worker_nodes_nsg_id : null
}


#--------------------------------------------------------------------------------
# STEP 3: Create the Bastion Host                                                #
#--------------------------------------------------------------------------------
module "bastion" {
  depends_on          = [module.vcn[0]]
  count               = var.is_bastion_created && var.is_vcn_created ? 1 : 0
  source              = "./modules/bastion"
  root_compartment_id = var.tenancy_ocid
  compartment_id      = var.compartment_id
  region              = var.region
  bastion_subnet_id   = module.vcn[0].bastion_public_subnet_id
  linux_images        = var.linux_images
  public_key_path     = var.ssh_public_key_path
  private_key_path    = var.ssh_private_key_path
  display_name        = "${var.display_name_prefix}-Bastion-Host"
  bastion_params      = var.bastion_params

  use_nsg        = var.use_nsg
  bastion_nsg_id = var.use_nsg ? module.vcn[0].bastion_nsg_id : null
}