##############################################################################################
# OKE Advanced Terraform Module: OKE Cluster and Node Pools                                  #
#                                                                                            #
# File: ~/oke_advanced_module/modules/oke/main.tf                                            #
#                                                                                            #
# Authors: Mahamat H. Guiagoussou, Payal Sharma, and Matthew McDaniel                        #
#                                                                                            #
# Copyright (c) 2025 Oracle                                                                  #
#                                                                                            #
#--------------------------------------------------------------------------------------------#
#                                                                                            #
# This Terraform file provisions the Oracle Kubernetes Engine (OKE) Cluster, Node Pools,     #
# and related networking/configuration resources.                                            #
#                                                                                            #
##############################################################################################


#********************************************************************************************#
# Data Source: Gathers available OKE node pool images                                        #
#********************************************************************************************#

# Used to compute dynamically the optimal OKE worker node image 
data "oci_containerengine_node_pool_option" "test_node_pool_option" {

    node_pool_option_id = "all"
    compartment_id = var.compartment_id

}


#********************************************************************************************#
# Cluster: Creates the primary OKE Kubernetes cluster                                        #
#********************************************************************************************#

resource "oci_containerengine_cluster" "this" {
  #Required
  compartment_id     = var.compartment_id
  kubernetes_version = var.control_plane_kubernetes_version
  name               = var.cluster_name
  vcn_id             = var.vcn_id


  #Optional
  cluster_pod_network_options {
    #Required
    cni_type = var.cni_type
  }


  endpoint_config {

    #Optional
    # Is Public IP Enabled for Cluster Control Plan Endpoint Config
    is_public_ip_enabled = var.control_plane_is_public
    # Use Network Security Group for Cluster Endpoint Config if flag is on
    #nsg_ids = var.use_nsg ? [var.k8api_nsg_id] : []
    # Kubernetest Endpoint Private Subnet OCID
    subnet_id = var.k8apiendpoint_private_subnet_id
  }


  # If you already have a defined tag use it
  #defined_tags = {"Operations.CostCenter"= "42"}


  # For simplicity we are exposing at least one free form tag
  freeform_tags = { "${var.cluster_freeform_tag_key}" = var.cluster_freeform_tag_value }


  image_policy_config {

    #Optional
    # Is Policy Enabled for Cluster Image Policy Config
    is_policy_enabled = var.image_signing_enabled
    dynamic "key_details" {
      for_each = var.image_signing_enabled == true ? toset(var.image_signing_key_id) : []
      content {
        kms_key_id = var.image_signing_key_id.value
        # Optional - Include if you want to use your own encryption key for image signing, else it's encrypted using Oracle-managed keys
        # kms_key_id = oci_kms_key.test_key.id
      }
    }
  }


  # Optional - use if you want to bring your own encryption key for k8 secrets, else it's encrypted using Oracle-managed keys
  #kms_key_id = oci_kms_key.test_key.id

  options {

    #Optional
    add_ons {

      #Optional
      # Cluster Options add_ons - is Kubernetes Dashboard Enabled
      is_kubernetes_dashboard_enabled = false
      # Cluster Options add_ons -  is Tiller enabled
      is_tiller_enabled = false
    }


    admission_controller_options {

      #Optional
      # Cluster Options Admission Controller options - is Pod Security Policy Enabled
      is_pod_security_policy_enabled = false
    }


    #ip_families = var.cluster_options_ip_families


    kubernetes_network_config {

      #Optional
      # Cluster Options Kubernetes Network Config Pods CIDR
      pods_cidr = "10.244.0.0/16" # CNI-assigned IP range
      # Cluster Options Kubernetes Network Config Services CIDR
      services_cidr = "10.96.0.0/16" # IP range for ClusterIP services
    }


    persistent_volume_config {
      #Optional
      #defined_tags = {"Operations.CostCenter"= "42"}
      freeform_tags = { "${var.cluster_freeform_tag_key}" = var.cluster_freeform_tag_value }

    }


    service_lb_config {

      #Optional
      #defined_tags = {"Operations.CostCenter"= "42"}
      freeform_tags = { "${var.cluster_freeform_tag_key}" = var.cluster_freeform_tag_value }
    }
    service_lb_subnet_ids = [var.serviceloadbalancers_public_subnet_id]
  }


  type = var.cluster_type #ENHANCED_CLUSTER or BASIC_CLUSTER
}


#********************************************************************************************#
# Node Pool: Provisions OKE worker node pools                                                #
#********************************************************************************************#

resource "oci_containerengine_node_pool" "this" {

  depends_on = [oci_containerengine_cluster.this]


  for_each = var.is_node_pools_created ? var.node_pools : {}


  # Required
  cluster_id     = oci_containerengine_cluster.this.id
  compartment_id = var.compartment_id
  name           = each.value.name
  node_shape     = each.value.shape


  dynamic "initial_node_labels" {
    for_each = each.value.node_labels
    content {
      key   = initial_node_labels.key
      value = initial_node_labels.value
    }
  }


  freeform_tags = { "${var.node_pool_freeform_tag_key}" = var.node_pool_freeform_tag_value }


  kubernetes_version = var.worker_nodes_kubernetes_version
  node_config_details {
    dynamic "placement_configs" {
      for_each = each.value.availability_domains
      content {
        availability_domain = placement_configs.value
        subnet_id           = var.workernodes_private_subnet_id
        #nsg_ids             = var.use_nsg ? [var.worker_nodes_nsg_id] : []
      }
    }


    # Added Tag inside the noe_config_details to propagate the tag to the worker node (on 09/30)
    freeform_tags = { "${var.node_pool_freeform_tag_key}" = var.node_pool_freeform_tag_value }
    

    size = each.value.number_of_nodes


    is_pv_encryption_in_transit_enabled = each.value.pv_in_transit_encryption
    node_pool_pod_network_option_details {
      cni_type       = var.cni_type
      pod_subnet_ids = var.is_cni_type_native ? [var.pods_private_subnet_id] : []
    }
  }


  node_pool_cycling_details {
    is_node_cycling_enabled = each.value.node_cycle_config.node_cycling_enabled
    maximum_surge           = each.value.node_cycle_config.maximum_surge
    maximum_unavailable     = each.value.node_cycle_config.maximum_unavailable
  }


  node_shape_config {
    memory_in_gbs = each.value.shape_config.memory
    ocpus         = each.value.shape_config.ocpus
  }


  node_source_details {
    image_id = [
      for images_list in data.oci_containerengine_node_pool_option.test_node_pool_option.sources :
      images_list if(
        !strcontains(lower(images_list.source_name), "-gpu-") &&                                                                                                        # Exclude GPU options
        !strcontains(lower(images_list.source_name), "-aarch64-") &&                                                                                                    # exclude ARM options
        strcontains(lower(images_list.source_name), format("-%s-", substr(var.worker_nodes_kubernetes_version, 1, length(var.worker_nodes_kubernetes_version) - 1))) && # only supported versions
        strcontains(lower(images_list.source_name), lower(each.value.operating_system))                                                                                 # include supported OS
      )
    ][0].image_id
    source_type             = "IMAGE" # each.value.source_type 
    boot_volume_size_in_gbs = each.value.boot_volume_size
  }


  ssh_public_key = file(each.value.ssh_key)
}

