################################################################################
# OKE Advanced Terraform Root Module                                           #
# File: ~/oke_advanced_module/variables.tf                                     #
#                                                                              #
# Description:                                                                 #
#   Input variables for advanced OKE (Oracle Kubernetes Engine) Terraform      #
#   deployments, with comments and validation for OCI, networking,             #
#   bastion, and cluster configuration.                                        #
#                                                                              #
# Authors: Mahamat H. Guiagoussou, Payal Sharma, Matthew McDaniel              #
# Copyright (c) 2025 Oracle                                                    #
################################################################################



#-------------------------------#
# Provider & Authentication     #
#-------------------------------#

# The OCID of your OCI tenancy where resources will be managed
variable "tenancy_ocid" {
  description = "Tenancy OCID for your Oracle Cloud tenancy."
  type        = string
  default     = "ocid1.tenancy.oc1..aaaaaaaa............"
}

# The OCID of the IAM user for Terraform operations
variable "user_ocid" {
  description = "User OCID used for managing resources in the tenancy."
  type        = string
}

# The fingerprint of the API signing key for Oracle Cloud authentication
variable "fingerprint" {
  description = "Fingerprint of the OCI API private key."
  type        = string
}

# Filesystem path to the OCI API private key used for signing API requests
variable "private_key_path" {
  description = "Local path to the OCI API private key file."
  type        = string
}

# The OCI region (e.g., us-ashburn-1) for resource creation
variable "region" {
  description = "OCI region identifier (e.g., us-ashburn-1)."
  type        = string
  default     = "us-ashburn-1"
}


#-------------------------------#
# Compartment Management        #
#-------------------------------#

# The OCID of the default compartment for OKE and associated resources
variable "compartment_id" {
  description = "Compartment OCID for OKE and related resources."
  type        = string
  default     = "ocid1.compartment.oc1..aaaaaaaa......."
}

# The OCID of the compartment dedicated to networking resources (VCN, subnets)
variable "networking_compartment_id" {
  description = "OCID for the compartment housing networking resources."
  type        = string
  default     = "ocid1.compartment.oc1..aaaaaaaa......"
}

# If true, use a dedicated compartment for networking; else, use the default
variable "is_networking_compartment_separate" {
  description = "Boolean to determine use of a separate networking compartment."
  type        = bool
  default     = true
}


#-------------------------------#
# Networking / VCN Configuration#
#-------------------------------#

# The primary CIDR block for the virtual cloud network (VCN)
variable "vcn_cidr_block" {
  description = "CIDR block assigned to the main VCN."
  type        = string
  default     = "10.0.0.0/16"
}

# String prefix for naming all resources (helps identify environment)
variable "display_name_prefix" {
  description = "Prefix for resource display names."
  type        = string
  default     = "OCI_OKE"
}

# String prefix used for hostnames in VCN and subnets
variable "host_name_prefix" {
  description = "Prefix for VM or resource hostnames."
  type        = string
  default     = "avcn"
}

# CIDR block for the private subnet that exposes the Kubernetes API endpoint
variable "k8apiendpoint_private_subnet_cidr_block" {
  description = "Subnet CIDR for private Kubernetes API endpoint."
  type        = string
  default     = "10.0.0.0/30"
}

# Subnet CIDR for OKE worker nodes (private subnet)
variable "workernodes_private_subnet_cidr_block" {
  description = "CIDR block for the OKE worker node private subnet."
  type        = string
  default     = "10.0.1.0/24"
}

# Subnet CIDR for the Kubernetes pods network
variable "pods_private_subnet_cidr_block" {
  description = "CIDR for pod network (private subnet)."
  type        = string
  default     = "10.0.32.0/19"
}

# Additional CIDR for pods (could be legacy/compatibility)
variable "pod_network_cidr" {
  description = "Legacy/compatibility pod network CIDR."
  type        = string
  default     = "10.0.1.0/24"
}

# Public subnet CIDR block for Kubernetes service load balancers
variable "serviceloadbalancers_public_subnet_cidr_block" {
  description = "CIDR block for public service load balancer subnet."
  type        = string
  default     = "10.0.2.0/24"
}

# Public subnet CIDR block for the bastion host
variable "bastion_public_subnet_cidr_block" {
  description = "CIDR for bastion host public subnet."
  type        = string
  default     = "10.0.3.0/24"
}


#-------------------------------#
# OKE Cluster Configuration     #
#-------------------------------#

# Version of Kubernetes to run on the OKE control plane
variable "control_plane_kubernetes_version" {
  description = "Kubernetes version for the OKE control plane."
  type        = string
  default     = "v1.33.1"
}

# Version of Kubernetes for all worker nodes (node pools)
variable "worker_nodes_kubernetes_version" {
  description = "Kubernetes version for OKE worker nodes."
  type        = string
  default     = "v1.33.1"
}

# Whether the OKE API endpoint (control plane) is public/private
variable "control_plane_is_public" {
  description = "If true, the OKE API endpoint is public."
  type        = bool
  default     = false
}

# CNI (container network interface) backend for this cluster
variable "cni_type" {
  description = "CNI type (e.g., OCI_VCN_IP_NATIVE or FLANNEL_OVERLAY)."
  type        = string
  default     = "OCI_VCN_IP_NATIVE"
}

# OKE cluster deployment type (Enhanced or Basic)
variable "cluster_type" {
  description = "OKE cluster type (ENHANCED_CLUSTER or BASIC_CLUSTER)."
  type        = string
  default     = "ENHANCED_CLUSTER"
}

# Enable container image signature verification for OKE
variable "image_signing_enabled" {
  description = "Enable image signing for container images."
  type        = bool
  default     = false
}

# The OCID of the KMS key for image signing (if enabled)
variable "image_signing_key_id" {
  description = "OCID for KMS key (if image signing enabled)."
  type        = string
  default     = ""
}

# Map (object) describing each OKE worker node pool
variable "worker_node_pools" {
  description = "Map of OKE worker node pool configurations."
  type = map(object({
    name  = string
    shape = string
    shape_config = object({
      memory = number
      ocpus  = number
    })
    boot_volume_size         = number
    operating_system         = string
    kubernetes_version       = string
    source_type              = string
    node_labels              = map(string)
    availability_domains     = list(string)
    number_of_nodes          = number
    pv_in_transit_encryption = bool
    node_cycle_config = object({
      node_cycling_enabled = bool
      maximum_surge        = number
      maximum_unavailable  = number
    })
    ssh_key = string
  }))
  default = {
    node_pools = {
      name  = "node_pool_1"
      shape = "VM.Standard.E5.Flex"
      shape_config = {
        memory = 16
        ocpus  = 1
      }
      boot_volume_size         = 50
      operating_system         = "Oracle-Linux"
      kubernetes_version       = "v1.33.1"
      source_type              = "IMAGE"
      node_labels              = { Trigger = "Nodes_Cycling_0" }
      availability_domains     = ["AQob:US-ASHBURN-AD-1", "AQob:US-ASHBURN-AD-2", "AQob:US-ASHBURN-AD-3"]
      number_of_nodes          = 1
      pv_in_transit_encryption = false
      node_cycle_config = {
        node_cycling_enabled = true
        maximum_surge        = 1
        maximum_unavailable  = 0
      }
      ssh_key = "oke_node_key.pub"
    }
  }
}


#-------------------------------#
# Bastion Host Configuration    #
#-------------------------------#

# Map of region to image/version OCIDs for compute/bastion/worker nodes
variable "linux_images" {
  description = "Map of Linux OS image OCIDs by region and version."
  type        = map(any)
  default = {
    us-ashburn-1 = {
      ol8_1_25_4 = "ocid1.image.oc1.iad.aaaaaaaaytaof66ujzy3wted7xjaicgshgb4mkp5q7fryvkvkkhmtdpd3tva"
      ol8_1_24_1 = "ocid1.image.oc1.iad.aaaaaaaaytaof66ujzy3wted7xjaicgshgb4mkp5q7fryvkvkkhmtdpd3tva"
    }
    us-phoenix-1 = {
      ol8_1_25_4 = "ocid1.image.oc1.phx.aaaaaaaalgvdp6hhnulo3tlxz3mtff625s7ix6ianpmv5l7chz5rcakrxbiq"
      ol8_1_24_1 = "ocid1.image.oc1.phx.aaaaaaaatghdqnhzrvrw334rfppfe6m7ibldsillwew3ogdkfgyihu74ra2q"
    }
  }
}

# Filesystem path to the public SSH key for compute, OKE, and bastion access
variable "ssh_public_key_path" {
  description = "Filesystem path to public SSH key for compute resources."
  type        = string
  default     = "oke_node_key.pub"
}

# Filesystem path to the private SSH key for secure access to compute resources
variable "ssh_private_key_path" {
  description = "Filesystem path to private SSH key for compute resources."
  type        = string
  default     = "oke_node_key"
}

# Configuration map for settings specific to the bastion instance
variable "bastion_params" {
  description = "Bastion host settings/configurations by logical group."
  type = map(object({
    ad_number    = number
    fault_domain = number
    shape        = string
    shape_config = object({
      memory = number
      ocpus  = number
    })
    hostname             = string
    version              = string
    boot_volume_size     = number
    assign_public_ip     = bool
    preserve_boot_volume = bool
    freeform_tags        = map(string)
    platform_config = object({
      is_symmetric_multi_threading_enabled = string
      type                                 = string
    })
    instance_options_legacy_imds_endpoints_disabled = bool
  }))
  default = {
    this_bastion = {
      ad_number    = 1
      fault_domain = 1
      shape        = "VM.Standard.E5.Flex"
      shape_config = { memory = 12, ocpus = 1 }
      platform_config = {
        is_symmetric_multi_threading_enabled = "true"
        type                                 = "AMD_VM"
      }
      hostname                                        = "tfokebh"
      version                                         = "ol8_1_25_4"
      boot_volume_size                                = 50
      assign_public_ip                                = true
      preserve_boot_volume                            = false
      instance_options_legacy_imds_endpoints_disabled = false
      freeform_tags = {
        project = "prod-network"
        env     = "production"
      }
    }
  }
}


#-------------------------------#
# Control Flags                 #
#-------------------------------#

# If true, the VCN and network resources will be created by the module
variable "is_vcn_created" {
  description = "Enable creation of VCN and networking resources."
  type        = bool
  default     = true
}

# If true, the OKE cluster will be created during apply
variable "is_k8cluster_created" {
  description = "Enable creation of OKE cluster during apply."
  type        = bool
  default     = true
}

# If true, one or more worker node pools will be created
variable "is_nodepool_created" {
  description = "Enable creation of OKE worker node pools."
  type        = bool
  default     = true
}

# If true, the Bastion host will be created
variable "is_bastion_created" {
  description = "Enable creation of bastion instance."
  type        = bool
  default     = true
}

# Use Network Security Groups (if false, uses Security Lists instead)
variable "use_nsg" {
  description = "If true, use NSGs instead of Security Lists for firewalls."
  type        = bool
  default     = false
}


#-------------------------------#
# Tagging and Misc              #
#-------------------------------#

# The freeform tag key for OKE cluster resources
variable "cluster_freeform_tag_key" {
  description = "Freeform tag key for OKE cluster resources."
  type        = string
  default     = "Environment"
}

# The freeform tag value for OKE cluster resources
variable "cluster_freeform_tag_value" {
  description = "Freeform tag value for OKE cluster resources."
  type        = string
  default     = "ORM Test/Deve"
}

# The freeform tag key for OKE node pool
variable "node_pool_freeform_tag_key" {
  description = "Freeform tag key for OKE node pools."
  type        = string
  default     = "LOB"
}

# The freeform tag value for OKE node pool
variable "node_pool_freeform_tag_value" {
  description = "Freeform tag value for OKE node pools."
  type        = string
  default     = "DevOps Tech with ORM"
}


#-------------------------------#
# Empty Variables (reserved)    #
#-------------------------------#

# RESERVED FOR FUTURE DNS LABEL OVERRIDES
variable "custom_dns_label" {
  description = "Custom DNS label for VCN or resources (leave empty for default behavior)."
  type        = string
  default     = null
}

# RESERVED FOR FUTURE TAGGING STRUCTURE
variable "custom_tags" {
  description = "Custom map of tags for resources (leave empty for default behavior)."
  type        = map(string)
  default     = null
}