##########################################################################################
# OKE Advanced Terraform Module: Bastion Host                                            #
#                                                                                        #
# File: ~/oke_advanced_module/modules/bastion/main.tf                                    #
#                                                                                        #
# Authors: Mahamat H. Guiagoussou, Payal Sharma, and Matthew McDaniel                    #
#                                                                                        #
# Copyright (c) 2025 Oracle                                                              #
#                                                                                        #
#----------------------------------------------------------------------------------------#
#                                                                                        #
# This file configures a bastion host (jump box) for secure SSH access to private nodes  #
# in your Oracle Cloud Infrastructure (OCI) Kubernetes Environment (OKE).                #
#                                                                                        #
# In order to access your private nodes with a public SSH key you will need to set up    #
# a bastion host (a.k.a. jump box).                                                      #
#                                                                                        #
# Learn more:                                                                            #
# https://docs.oracle.com/en-us/iaas/Content/ContEng/Tasks/contengsettingupbastion.htm   #
#                                                                                        #
##########################################################################################


#****************************************************************************************#
# Fetches the list of Availability Domains in your tenancy                               #
#****************************************************************************************#

data "oci_identity_availability_domains" "ads" {
  #Required
  compartment_id = var.root_compartment_id # Tenancy is the root compartment.
}


#****************************************************************************************#
# Creates the Bastion Host VM Instance                                                   #
#****************************************************************************************#

resource "oci_core_instance" "this" {
  for_each = var.bastion_params

  compartment_id      = var.compartment_id
  availability_domain = data.oci_identity_availability_domains.ads.availability_domains[each.value.ad_number - 1].name
  shape               = each.value.shape
	shape_config {
		memory_in_gbs = each.value.shape_config.memory   # "12"
		ocpus = each.value.shape_config.ocpus  # "1"
	}
  display_name         = var.display_name
  preserve_boot_volume = each.value.preserve_boot_volume
  fault_domain         = format("FAULT-DOMAIN-%d", each.value.fault_domain)
  freeform_tags        = each.value.freeform_tags

  is_pv_encryption_in_transit_enabled = "true"

  instance_options {
    are_legacy_imds_endpoints_disabled = each.value.instance_options_legacy_imds_endpoints_disabled #"false"
  }

  platform_config {
    is_symmetric_multi_threading_enabled = each.value.platform_config.is_symmetric_multi_threading_enabled  #"true"
    type                                 = each.value.platform_config.type  #"AMD_VM"
  }

  create_vnic_details {
    assign_ipv6ip             = "false"
    assign_private_dns_record = "true"
    assign_public_ip          = each.value.assign_public_ip
    subnet_id                 = var.bastion_subnet_id
    nsg_ids = var.use_nsg ? [var.bastion_nsg_id] : []
  }


  source_details {
    boot_volume_size_in_gbs = each.value.boot_volume_size
    source_type             = "image"
    source_id               =  var.linux_images[var.region][each.value.version] # data.oci_core_images.linux_image.images[0].id
  }

  metadata = {
    ssh_authorized_keys = file(var.public_key_path)
  }

  lifecycle {
    ignore_changes = [create_vnic_details]
  }

}