##############################################################################################
# OKE Advanced Terraform Root Module: VCN and Subnet Networking                              #
#                                                                                            #
# File: ~/oke_advanced_module/modules/vcn/main.tf                                            #
#                                                                                            #
# Authors: Mahamat H. Guiagoussou, Payal Sharma, and Matthew McDaniel                        #
#                                                                                            #
# Copyright (c) 2025 Oracle                                                                  #
#                                                                                            #
#--------------------------------------------------------------------------------------------#
#                                                                                            #
# This file provisions and configures OCI VCN, gateways, subnets, and related networking     #
# components as required for advanced OKE environments.                                      #
#                                                                                            #
##############################################################################################


#*************************** VCN: Virtual Cloud Network Resource ****************************#

resource "oci_core_vcn" "this" {
  compartment_id = var.compartment_id
  cidr_block     = var.vcn_cidr_block
  display_name   = "${var.display_name_prefix}-VCN"
  dns_label      = var.host_name_prefix
}


#******************************** Internet Gateway Resource *********************************#

resource "oci_core_internet_gateway" "this" {
  compartment_id = var.compartment_id
  display_name   = "${var.display_name_prefix}-Internet-Gateway"
  enabled        = "true"
  vcn_id         = oci_core_vcn.this.id
}


#*********************************** NAT Gateway Resource ***********************************#

resource "oci_core_nat_gateway" "this" {
  compartment_id = var.compartment_id
  display_name   = "${var.display_name_prefix}-NAT-Gateway"  
  vcn_id         = oci_core_vcn.this.id
}


#******************************* Service Gateway & All OCI Services *************************#

data "oci_core_services" "all_oci_services" {
  filter {
    name   = "name"
    values = ["All .* Services In Oracle Services Network"]
    regex  = true
  }
}


#************************************ Service Gateway ***************************************#

resource "oci_core_service_gateway" "this" {
  compartment_id = var.compartment_id
  services {
    #service_id = data.oci_core_services.all_oci_services.services.id
    service_id = lookup(data.oci_core_services.all_oci_services.services[0], "id")
  }
  display_name   = "${var.display_name_prefix}-Service-Gateway"
  vcn_id         = oci_core_vcn.this.id 
}



#************************************** Subnet Resources ************************************#

##############################################################################################
# SUBNETS - This is a Structired Networking Mobule. It is composed of the following subnets: #  
#                                                                                            #                                               #
#   Subnet 1: Private Subnet for K8 API Endpoint Subnet                                      #
#   Subnet 2: Private Subnet For Worker Nodes                                                #
#   Subnet 3: Private Subnet For Pods                                                        # 
#   Subnet 4: Public Subnet For Load Balancers                                               #
#   Subnet 5: Public Subnet for Bastion Host                                                 #
#                                                                                            #
#  For each Subnet the code includes:                                                        #
#     (1) Subnet definition (Private or Public)                                              #
#     (2) Route Table (shared for worker nodes and pods)                                     #
#     (3) Security List (these are examples only)                                            #
#     (4) NSG (Network Security Group), out of Scope                                         #             
##############################################################################################



#--------------------------------------------------------------------------------------------#
# Subnet 1: Private Subnet for K8 API Endpoint Subnet                                        #
#--------------------------------------------------------------------------------------------#

resource "oci_core_subnet" "Private-Subnet-For-K8-API-Endpoint" {
  cidr_block        = var.k8apiendpoint_private_subnet_cidr_block
  compartment_id    = var.compartment_id
  dhcp_options_id = oci_core_vcn.this.default_dhcp_options_id
  display_name    = "${var.display_name_prefix}-Private-Subnet-For-Kubernetes-API-Endpoint"
  dns_label       = "${var.host_name_prefix}k8pnetep"
    ipv6cidr_blocks = [
  ]
  prohibit_internet_ingress  = "true"
  prohibit_public_ip_on_vnic = "true"
  route_table_id    = oci_core_route_table.Route-Table-For-Private-K8-API-Endpoint-Subnet.id
  security_list_ids = var.use_nsg ? [] : [oci_core_security_list.Security-List-For-K8-APIendpoint.id]
  vcn_id            = oci_core_vcn.this.id
}


# Routing Table For Private K8 API Endpoint Subnet
resource "oci_core_route_table" "Route-Table-For-Private-K8-API-Endpoint-Subnet" {

  compartment_id = var.compartment_id
  vcn_id         = oci_core_vcn.this.id

  display_name = "${var.display_name_prefix}-RoutingTable-For-Private-K8-API-Endpoint-Subnet"
  route_rules {
    description       = "Route Table for ${var.display_name_prefix} For Private K8 API Endpoint Subnet through NAT"
    destination       = "0.0.0.0/0"
    destination_type  = "CIDR_BLOCK"
    network_entity_id = oci_core_nat_gateway.this.id
  }


  route_rules {
    description       = "Route Table for ${var.display_name_prefix} For Private K8 API Endpoint Subnet through SGW"
    destination       = "all-iad-services-in-oracle-services-network"
    destination_type  = "SERVICE_CIDR_BLOCK"
    network_entity_id = oci_core_service_gateway.this.id
  }
}


# Security List For K8 API Endpoint
resource "oci_core_security_list" "Security-List-For-K8-APIendpoint" {

  compartment_id = var.compartment_id
  vcn_id         = oci_core_vcn.this.id

  display_name = "${var.display_name_prefix}-Security-List-For-K8-API-Endpoint"


  # ------------- #
  # Ingress Rules
  # --------------#

  ingress_security_rules {
    description = "Kubernetes 4 to Kubernetes API Endpoint communication."
    protocol    = "6"
    source      = var.workernodes_private_subnet_cidr_block # e.g.: 10.0.1.0/24
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "6443"
      min = "6443"
    }
  }


  ingress_security_rules {
    description = " 	Kubernetes worker to control plane communication."
    protocol    = "6"
    source      = var.workernodes_private_subnet_cidr_block # e.g.: 10.0.1.0/24
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "12250"
      min = "12250"
    }
  }


  ingress_security_rules {
    description = "Path Discovery."
    icmp_options {
      code = "4"
      type = "3"
    }
    protocol    = "1"
    source      = var.workernodes_private_subnet_cidr_block # e.g.: 10.0.1.0/24
    source_type = "CIDR_BLOCK"
    stateless   = "false"
  }


  ingress_security_rules {
    description = "Kubernetes worker to Kubernetes API Endpoint communication."
    protocol    = "6"
    source      = var.workernodes_private_subnet_cidr_block
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "6443"
      min = "6443"
    }
  }


  ingress_security_rules {
    description = "Bastion host to Kubernetes API Endpoint communication."
    protocol    = "6"
    source      = var.bastion_public_subnet_cidr_block # e.g.: 10.0.3.0/24
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "6443"
      min = "6443"
    }
  }


  # ------------ #
  # Egress Rules #
  # ------------ #


  egress_security_rules {
    description      = "Allow Kubernetes control plane to communicate with OKE."
    destination      = "all-iad-services-in-oracle-services-network"
    destination_type = "SERVICE_CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
  }


  egress_security_rules {
    description = "Path Discovery."
    icmp_options {
      code = "4"
      type = "3"
    }
    destination      = "all-iad-services-in-oracle-services-network"
    destination_type = "SERVICE_CIDR_BLOCK"
    protocol         = "1"
    stateless        = "false"
  }


  egress_security_rules {
    description      = " Allow Kubernetes control plane to communicate with worker nodes."
    destination      = var.workernodes_private_subnet_cidr_block # e.g.: 10.0.1.0/24
    destination_type = "CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
  }


  egress_security_rules {
    description = "Path Discovery."
    icmp_options {
      code = "4"
      type = "3"
    }
    destination      = var.workernodes_private_subnet_cidr_block # e.g.: 10.0.1.0/24
    destination_type = "CIDR_BLOCK"
    protocol         = "1"
    stateless        = "false"
  }

}



#--------------------------------------------------------------------------------------------#
# Subnet 2: Private Subnet For Worker Nodes                                                  #
#--------------------------------------------------------------------------------------------#

resource "oci_core_subnet" "Private-Subnet-For-Worker-Nodes" {

  cidr_block                 = var.workernodes_private_subnet_cidr_block
  compartment_id             = var.compartment_id
  dhcp_options_id            = oci_core_vcn.this.default_dhcp_options_id
  display_name               = "${var.display_name_prefix}-Private-Subnet-For-Worker-Nodes"
  dns_label                  = "${var.host_name_prefix}wnprvnet"
  ipv6cidr_blocks            = []
  prohibit_internet_ingress  = "true"
  prohibit_public_ip_on_vnic = "true"
  route_table_id             = oci_core_route_table.Route-Table-For-Private-Subnet-For-Worker-Nodes.id
  security_list_ids = [
    oci_core_security_list.Security-List-For-Private-Subnet-For-Worker-Nodes.id
  ]
  vcn_id = oci_core_vcn.this.id
}


# Routing Table For Private Subnet For Worker Nodes 
resource "oci_core_route_table" "Route-Table-For-Private-Subnet-For-Worker-Nodes" {

  compartment_id = var.compartment_id
  vcn_id         = oci_core_vcn.this.id

  display_name = "${var.display_name_prefix}-RoutingTable-For-Private-Subnet-For-Worker-Nodes"

  route_rules {
    description       = "Route Table for ${var.display_name_prefix} Route Rule 1 For Private Subnet For Worker Nodes"
    destination       = "0.0.0.0/0"
    destination_type  = "CIDR_BLOCK"
    network_entity_id = oci_core_nat_gateway.this.id
  }

  route_rules {
    description       = "Route Table for ${var.display_name_prefix} Route Rule 2 For Private K8 APIEndpoint Subnet through SGW"
    destination       = "all-iad-services-in-oracle-services-network"
    destination_type  = "SERVICE_CIDR_BLOCK"
    network_entity_id = oci_core_service_gateway.this.id
  }

}


# Private Subnet Security List For Private Subnet For Worker Nodes
resource "oci_core_security_list" "Security-List-For-Private-Subnet-For-Worker-Nodes" {

  compartment_id = var.compartment_id
  vcn_id         = oci_core_vcn.this.id

  display_name = "${var.display_name_prefix}-Security-List-For-Private-Subnet-For-Worker-Nodes"


  # ------------- #
  # Ingress Rules #
  # ------------- #

  ingress_security_rules {
    description = "Allow pods on one worker node to communicate with pods on other worker nodes."
    protocol    = "all"
    source      = var.workernodes_private_subnet_cidr_block # e.g.: 10.0.1.0/24
    source_type = "CIDR_BLOCK"
    stateless   = "false"
  }


  ingress_security_rules {
    description = "Allow Kubernetes control plane to communicate with worker nodes."
    #icmp_options = <<Optional value not found in discovery>>
    protocol    = "6"
    source      = var.k8apiendpoint_private_subnet_cidr_block # "10.0.0.0/30"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
  }


  ingress_security_rules {
    description = "Path Discovery."
    icmp_options {
      code = "4"
      type = "3"
    }
    protocol    = "1"
    source      = "0.0.0.0/0"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
  }


  ingress_security_rules {
    description = "Allow inbound SSH traffic to managed nodes."
    protocol    = "6"
    source      = var.bastion_public_subnet_cidr_block # "10.0.3.0/24"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "22"
      min = "22"
    }
  }


  ingress_security_rules {
    description = "Load balancer to worker nodes node ports."
    protocol    = "6"
    source      = var.serviceloadbalancers_public_subnet_cidr_block # "10.0.2.0/24"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "32767"
      min = "30000"
    }
  }


  ingress_security_rules {
    description = "Load balancer to worker nodes node ports."
    protocol    = "17"
    source      = var.serviceloadbalancers_public_subnet_cidr_block # "10.0.2.0/24"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    udp_options {
      max = "32767"
      min = "30000"
    }
  }


  ingress_security_rules {
    description = "Allow load balancer to communicate with kube-proxy on worker nodes."
    protocol    = "6"
    source      = var.serviceloadbalancers_public_subnet_cidr_block # "10.0.2.0/24"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "10256"
      min = "10256"
    }
  }


  ingress_security_rules {
    description = "Allow load balancer to communicate with kube-proxy on worker nodes."
    protocol    = "17"
    source      = var.serviceloadbalancers_public_subnet_cidr_block # "10.0.2.0/24"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    udp_options {
      max = "10256"
      min = "10256"
    }
  }


  # ------------ #
  # Egress Rules #
  # ------------ #

  egress_security_rules {
    description      = "Allow pods on one worker node to communicate with pods on other worker nodes."
    destination      = var.workernodes_private_subnet_cidr_block # e.g.: 10.0.1.0/24
    destination_type = "CIDR_BLOCK"
    protocol         = "all"
    stateless        = "false"
  }


  egress_security_rules {
    description      = "Allow worker nodes to communicate with OKE."
    destination      = "all-iad-services-in-oracle-services-network"
    destination_type = "SERVICE_CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
  }


  egress_security_rules {
    description      = "Kubernetes worker to Kubernetes API endpoint communication (TCP)."
    destination      = var.k8apiendpoint_private_subnet_cidr_block # e.g.: 10.0.0.0/30
    destination_type = "CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
    tcp_options {
      max = "6443"
      min = "6443"
    }
  }


  egress_security_rules {
    description      = "Kubernetes worker to Kubernetes API endpoint communication (UDP)."
    destination      = var.k8apiendpoint_private_subnet_cidr_block # e.g.: 10.0.0.0/30
    destination_type = "CIDR_BLOCK"
    protocol         = "17"
    stateless        = "false"
    udp_options {
      max = "6443"
      min = "6443"
    }
  }


  egress_security_rules {
    description      = "Kubernetes worker to control plane communication (TCP)."
    destination      = var.k8apiendpoint_private_subnet_cidr_block # e.g.: 10.0.0.0/30
    destination_type = "CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
    tcp_options {
      max = "12250"
      min = "12250"
    }
  }


  egress_security_rules {
    description      = "Kubernetes worker to control plane communication (UDP)."
    destination      = var.k8apiendpoint_private_subnet_cidr_block # e.g.: 10.0.0.0/30
    destination_type = "CIDR_BLOCK"
    protocol         = "17"
    stateless        = "false"
    udp_options {
      max = "12250"
      min = "12250"
      #source_port_range = <<Optional value not found in discovery>>
    }
  }


  egress_security_rules {
    description      = "Allow worker nodes to communicate with internet."
    destination      = "0.0.0.0/0"
    destination_type = "CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
  }

}



#--------------------------------------------------------------------------------------------#
# Subnet 3: Private Subnet For Pods                                                          # 
#--------------------------------------------------------------------------------------------#

resource "oci_core_subnet" "Private-Subnet-For-Pods" {

  cidr_block                 = var.pods_private_subnet_cidr_block
  compartment_id             = var.compartment_id
  dhcp_options_id            = oci_core_vcn.this.default_dhcp_options_id
  display_name               = "${var.display_name_prefix}-Private-Subnet-For-Pods"
  dns_label                  = "${var.host_name_prefix}pprvnet"
  ipv6cidr_blocks            = []
  prohibit_internet_ingress  = "true"
  prohibit_public_ip_on_vnic = "true"
  route_table_id             = oci_core_route_table.Route-Table-For-Private-Subnet-For-Worker-Nodes.id
  security_list_ids = [
    oci_core_security_list.Security-List-For-Private-Subnet-For-Pods.id
  ]
  vcn_id = oci_core_vcn.this.id
}


# Routing Table For Private Subnet For Node Pods 
# - Reuse the Worker Nodes Private Subnet Route Table 
# - Same definition as the Worker Nodes Route Table


# Private Subnet Security List For Private Subnet For Worker Nodes
resource "oci_core_security_list" "Security-List-For-Private-Subnet-For-Pods" {

  compartment_id = var.compartment_id
  vcn_id         = oci_core_vcn.this.id

  display_name = "${var.display_name_prefix}-Security-List-For-Private-Subnet-For-Pods"


  # ------------- #
  # Ingress Rules #
  # ------------- #


  ingress_security_rules {
    description = "Allow Kubernetes control plane to communicate with worker nodes."
    protocol    = "all"
    source      = var.k8apiendpoint_private_subnet_cidr_block # "10.0.0.0/30"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
  }


  ingress_security_rules {
    description = "Allow worker node to access pods."
    protocol    = "all"
    source      = var.workernodes_private_subnet_cidr_block # e.g.: 10.0.1.0/24
    source_type = "CIDR_BLOCK"
    stateless   = "false"
  }


  ingress_security_rules {
    description = "Allow Kubernetes API Endpoint to communicate with pods."
    protocol    = "all"
    source      = var.pods_private_subnet_cidr_block # "10.0.0.0/19"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
  }


  # ------------ #
  # Egress Rules #
  # ------------ #


  # Allow pods to communicate with other pods.
  egress_security_rules {
    description      = "Allow pods on one worker node to communicate with pods on other worker nodes."
    destination      = var.pods_private_subnet_cidr_block # e.g.: 10.0.1.0/19
    destination_type = "CIDR_BLOCK"
    protocol         = "all"
    stateless        = "false"
  }

  
  # Path discovery
  egress_security_rules {
    description = "Path Discovery."
    icmp_options {
      code = "4"
      type = "3"
    }
    destination      = "all-iad-services-in-oracle-services-network"
    destination_type = "SERVICE_CIDR_BLOCK"
    protocol         = "1"
    stateless        = "false"
  }
  

  # Allow pods to communicate with OCI services.  
  egress_security_rules {
    description      = "Allow worker nodes to communicate with OCI services."
    destination      = "all-iad-services-in-oracle-services-network"
    destination_type = "SERVICE_CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
  }


  # (optional) Allow pods to communicate with internet.
  egress_security_rules {
    description      = "Allow node pods to communicate with internet."
    destination      = "0.0.0.0/0"
    destination_type = "CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
    tcp_options {
      max = "443"
      min = "443"
    }
  }


  egress_security_rules {
    description      = "Kubernetes worker to Kubernetes API endpoint communication (TCP)."
    destination      = var.k8apiendpoint_private_subnet_cidr_block # e.g.: 10.0.0.0/30
    destination_type = "CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
    tcp_options {
      max = "6443"
      min = "6443"
    }
  }


  egress_security_rules {
    description      = "Kubernetes worker to control plane communication (TCP)."
    destination      = var.k8apiendpoint_private_subnet_cidr_block # e.g.: 10.0.0.0/30
    destination_type = "CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
    tcp_options {
      max = "12250"
      min = "12250"
    }
  }

}



#--------------------------------------------------------------------------------------------#
# Subnet 4: Regional Public Subnet for Service Load Balancers                                #                            
#--------------------------------------------------------------------------------------------#

resource "oci_core_subnet" "Public-Subnet-For-Load-Balancers" {

  cidr_block                 = var.serviceloadbalancers_public_subnet_cidr_block
  compartment_id             = var.compartment_id
  dhcp_options_id            = oci_core_vcn.this.default_dhcp_options_id
  display_name               = "${var.display_name_prefix}-Public-Subnet-For-Load-Balancers"
  dns_label                  = "${var.host_name_prefix}slbpubnet"
  ipv6cidr_blocks            = []
  prohibit_internet_ingress  = "false"
  prohibit_public_ip_on_vnic = "false"
  route_table_id             = oci_core_route_table.Route-Table-For-Public-Load-Balancers-Subnet.id
  security_list_ids = [
    oci_core_security_list.Security-List-For-Public-Load-Balancers-Subnet.id,
  ]
  vcn_id = oci_core_vcn.this.id
}


# Routing Table For Public Load Balancers Subnet
resource "oci_core_route_table" "Route-Table-For-Public-Load-Balancers-Subnet" {

  compartment_id = var.compartment_id
  vcn_id         = oci_core_vcn.this.id

  display_name = "${var.display_name_prefix}-RoutingTable-For-Public-Load-Balancers-Subnet"
  route_rules {
    description       = "Route Table for ${var.display_name_prefix} Route Rule For Public Load Balancers Subnet"
    destination       = "0.0.0.0/0"
    destination_type  = "CIDR_BLOCK"
    network_entity_id = oci_core_internet_gateway.this.id
  }
}


# Security List for Public Load Balancers Subnet. 

resource "oci_core_security_list" "Security-List-For-Public-Load-Balancers-Subnet" {

  compartment_id = var.compartment_id
  vcn_id         = oci_core_vcn.this.id

  display_name = "${var.display_name_prefix}-Security-List-For-Public-Load-Balancers-Subnet"

  # ------------- #
  # Ingress Rules #
  # ------------- #

  ingress_security_rules {
    description = "Load balancer listener protocol and port. Customize as required (e.g.: TCP from Internet) on port 443"
    protocol    = "6"
    source      = "0.0.0.0/0"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "443"
      min = "443"
      source_port_range {
        max = "443"
        min = "443"
      }
    }
  }


  ingress_security_rules {
    description = "Load balancer listener protocol and port. Customize as required (e.g.: TCP from Internet) on port 8080)"
    protocol    = "6"
    source      = "0.0.0.0/0"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "8080"
      min = "8080"
      source_port_range {
        max = "8080"
        min = "8080"
      }
    }
  }


  ingress_security_rules {
    description = "Load balancer listener protocol and port 80. Customize as required (e.g.: TCP from Internet)"
    protocol    = "6"
    source      = "0.0.0.0/0"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "80"
      min = "80"
      source_port_range {
        max = "80"
        min = "80"
      }
    }
  }


  ingress_security_rules {
    protocol    = "6"
    source      = "0.0.0.0/0"
    source_type = "CIDR_BLOCK"
    stateless   = false
    tcp_options {
      max = 80
      min = 80
    }
  }


  ingress_security_rules {
    description = "Load balancer listener protocol and port. Customize as required (e.g.: UDP from Internet) on port 443"
    protocol    = "17"
    source      = "0.0.0.0/0"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    udp_options {
      max = "443"
      min = "443"
      source_port_range {
        max = "443"
        min = "443"
      }
    }
  }


  # ------------ #
  # Egress Rules #
  # ------------ #

  egress_security_rules {
    description      = "Load balancer to worker nodes node ports."
    destination      = var.workernodes_private_subnet_cidr_block # "10.0.1.0/24"
    destination_type = "CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
    tcp_options {
      max = "32767"
      min = "30000"
    }
  }


  egress_security_rules {
    description      = "Allow load balancer to communicate with kube-proxy on worker nodes. [TCP Port Range: 30000-32767]"
    destination      = var.workernodes_private_subnet_cidr_block # "10.0.1.0/24"
    destination_type = "CIDR_BLOCK"
    protocol         = "6"
    stateless        = "false"
    tcp_options {
      max = "10256"
      min = "10256"
    }
  }


  egress_security_rules {
    description      = "Allow load balancer to communicate with kube-proxy on worker nodes."
    destination      = var.workernodes_private_subnet_cidr_block # "10.0.1.0/24"
    destination_type = "CIDR_BLOCK"
    protocol         = "17"
    stateless        = "false"
    udp_options {
      max = "32767"
      min = "30000"
    }
  }


  egress_security_rules {
    description      = "Allow load balancer to communicate with kube-proxy on worker nodes."
    destination      = var.workernodes_private_subnet_cidr_block # "10.0.1.0/24"
    destination_type = "CIDR_BLOCK"
    protocol         = "17"
    stateless        = "false"
    udp_options {
      max = "10256"
      min = "10256"
    }
  }

  lifecycle {
    ignore_changes = [ingress_security_rules, egress_security_rules]
  }

}



#--------------------------------------------------------------------------------------------#
# Subnet 5: Regional Public Subnet for Bastion Host.                                         #
#--------------------------------------------------------------------------------------------#

resource "oci_core_subnet" "Public-Subnet-For-Bastion-Host" {

  #availability_domain = <<Optional value not found in discovery>>
  cidr_block                 = var.bastion_public_subnet_cidr_block
  compartment_id             = var.compartment_id
  dhcp_options_id            = oci_core_vcn.this.default_dhcp_options_id
  display_name               = "${var.display_name_prefix}-Public-Subnet-For-Bastion-Host"
  dns_label                  = "${var.host_name_prefix}btpnet"
  ipv6cidr_blocks            = []
  prohibit_internet_ingress  = "false"
  prohibit_public_ip_on_vnic = "false"
  route_table_id             = oci_core_vcn.this.default_route_table_id
  security_list_ids = [
    oci_core_vcn.this.default_security_list_id,
  ]
  vcn_id = oci_core_vcn.this.id
}


# Default Routing Table
resource "oci_core_default_route_table" "Default-Route-Table" {

  compartment_id             = var.compartment_id
  display_name               = "${var.display_name_prefix}-Default-RoutingTable"
  manage_default_resource_id = oci_core_vcn.this.default_route_table_id
  route_rules {
    description       = "Route Table for ${var.display_name_prefix} Default Route Table"
    destination       = "0.0.0.0/0"
    destination_type  = "CIDR_BLOCK"
    network_entity_id = oci_core_internet_gateway.this.id
  }
}


# Default DHCP Options 
resource "oci_core_default_dhcp_options" "Default-DHCP-Options" {

  compartment_id             = var.compartment_id
  display_name               = "${var.display_name_prefix}-Default-DHCP-Options"
  domain_name_type           = "CUSTOM_DOMAIN"
  manage_default_resource_id = oci_core_vcn.this.default_dhcp_options_id
  options {
    custom_dns_servers = []
    server_type        = "VcnLocalPlusInternet"
    type               = "DomainNameServer"
  }
  options {
    search_domain_names = [
      "${var.host_name_prefix}core.oraclevcn.com"
    ]
    type = "SearchDomain"
  }
}


# Default Security List 
resource "oci_core_default_security_list" "Default-Security-List" {

  compartment_id = var.compartment_id

  display_name = "${var.display_name_prefix}-Default-Security-List"

  egress_security_rules {
    description      = "Egress Open to all protocols"
    destination      = "0.0.0.0/0"
    destination_type = "CIDR_BLOCK"

    protocol  = "all"
    stateless = "false"
  }


  ingress_security_rules {
    #description = <<Optional value not found in discovery>>
    icmp_options {
      code = "4"
      type = "3"
    }
    protocol    = "1"
    source      = "0.0.0.0/0"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
  }
  

  ingress_security_rules {
    #description = <<Optional value not found in discovery>>
    icmp_options {
      code = "-1"
      type = "3"
    }
    protocol    = "1"
    source      = "10.0.0.0/16"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
  }


  ingress_security_rules {
    #description = <<Optional value not found in discovery>>
    protocol    = "6"
    source      = "0.0.0.0/0"
    source_type = "CIDR_BLOCK"
    stateless   = "false"
    tcp_options {
      max = "22"
      min = "22"
    }
  }

  manage_default_resource_id = oci_core_vcn.this.default_security_list_id
}


