#
# oci-email-send-python version 1.0.
#
# Copyright (c) 2020 Oracle, Inc.
# Licensed under the Universal Permissive License v 1.0 as shown at https://oss.oracle.com/licenses/upl.
#

import io
import json
import smtplib 
import email.utils
from email.mime.text import MIMEText
from email.message import EmailMessage
import os
import sys
import oci
import psycopg2
import base64

from fdk import response

def sendEmail(subject, sender_name, sender_email, recipient, body, smtp_host, smtp_port, smtp_password, smtp_username):
    msg            = EmailMessage()
    msg['Subject'] = subject
    msg['From']    = email.utils.formataddr((sender_name, sender_email))
    msg['To']      = recipient
    msg.set_content(body, subtype='html')

    server = smtplib.SMTP(smtp_host, smtp_port)
    server.ehlo()
    server.starttls()
    server.login(smtp_username, smtp_password)
    server.send_message(msg)
    server.quit()

def decodeSecret(p_signer,p_secretName,p_vaultOCID):
    secretClient    = oci.secrets.SecretsClient(config={}, signer=p_signer)
    secret          = secretClient.get_secret_bundle_by_name(secret_name=p_secretName,vault_id=p_vaultOCID).data
    secret_content  = secret.secret_bundle_content.content.encode("utf-8")
    decodedSecret   = base64.b64decode(secret_content).decode("utf-8")
    return decodedSecret

def handler(ctx, data: io.BytesIO=None):
    smtp_username = smtp_password = smtp_host = ""
    smtp_port = 0
    sender_email = sender_name = recipient = subject = body = ""
    signer = oci.auth.signers.get_resource_principals_signer()

    # OCI Object Storage configuration
    BUCKET_NAME = "templates_bucket"
    OBJECT_KEY  = "email_template.html"
    
    # Get SMTP password from Vault
    try:
        cfg           = ctx.Config()
        secretName1   = cfg["smtp-password"] # this is a secret in the vault
        secretName2   = cfg["smtp-username"] # this is a secret in the vault
        secretName3   = cfg["db-password"] # this is a secret in the vault
        secretName4   = cfg["db-username"] # this is a secret in the vault
        vaultOcid     = cfg["vault-ocid"]
        body          = json.loads(data.getvalue())
        smtp_password = decodeSecret(p_signer=signer,p_secretName=secretName1,p_vaultOCID=vaultOcid)
        smtp_username = decodeSecret(p_signer=signer,p_secretName=secretName2,p_vaultOCID=vaultOcid)
        db_password   = decodeSecret(p_signer=signer,p_secretName=secretName3,p_vaultOCID=vaultOcid)
        db_username   = decodeSecret(p_signer=signer,p_secretName=secretName4,p_vaultOCID=vaultOcid)
        
    except Exception:
        raise 

  
    # Get other values from context
    try:
        cfg       = ctx.Config()
        smtp_host = cfg["smtp-host"]
        smtp_port = cfg["smtp-port"]
    except Exception as ex:
        print('ERROR: Missing configuration key', ex, flush=True)
        raise

    object_storage = oci.object_storage.ObjectStorageClient(config={}, signer=signer)
    namespace      = object_storage.get_namespace().data

    # Download email template from Object Storage
    try:
        get_object_response = object_storage.get_object(
            namespace_name = namespace,
            bucket_name    = BUCKET_NAME,
            object_name    = OBJECT_KEY
        )
        email_template = get_object_response.data.content.decode()  # Decode the content to string
    except Exception as e:
        print('ERROR: Getting object from bucket', ex, flush=True)
        raise

    try:
        payload_bytes = data.getvalue()
        if payload_bytes==b'':
            raise KeyError('No keys in payload')
        payload       = json.loads(payload_bytes)
        sender_email  = payload["sender-email"]
        sender_name   = payload["sender-name"]
        subject       = payload["subject"]
        body          = email_template          
    except Exception as ex:
        print('ERROR: Missing key in payload', ex, flush=True)
        raise
    

    # Connect to PostgreSQL database
    try:
        conn = psycopg2.connect(
            dbname   = 'postgres', #DB_NAME
            user     = db_username,
            password = db_password, 
            host     = '...', #to do - place here the connection endpoint IP
            port     = '5432' #DB_PORT
        )
        cursor = conn.cursor()
    except psycopg2.Error as e:
        print(f"Error connecting to PostgreSQL database: {e}")
        exit()
   
    # Query to retrieve email addresses from the table
    query = "SELECT address FROM emails;"

    try:
        cursor.execute(query)
        rows       = cursor.fetchall()
        email_list = []
        for row in rows:
            recipient = row[0]  
            email_list.append(recipient)
            sendEmail(subject, sender_name, sender_email, recipient, body, smtp_host, smtp_port, smtp_password, smtp_username)
    except psycopg2.Error as e:
        print(f"Error executing query: {e}")
        conn.close()
        exit()


    return f"Email successfully sent to {email_list}!"
